// Copyright 2025 by Carnegie Mellon University
// See license information in LICENSE.txt

package org.cert.netsa.io.silk

import enumeratum.values.{ShortEnum, ShortEnumEntry}

/** A SiLK bag data type. May be converted to and from [[scala.Short Short]] values.
  *
  * @param value The short value representing this bag data type.
  *
  * @see [[BagDataType$ the companion object]] for more details
  * @see [[BagReader]]
  */
sealed abstract class BagDataType(val value: Short) extends ShortEnumEntry

case object BagDataType extends ShortEnum[BagDataType] {

  /** Collection of all known valid SiLK bag data types.
    * @group Members
    */
  lazy val values = findValues
  // Note: lazy to avoid cyclic initialization problem

  /** Converts a short value to a bag data (field) type.
    * @return The data type represented by this value.
    * @throws SilkDataFormatException if the short represents no known data type.
    * @group Members
    */
  def apply(v: Short): BagDataType =
    withValueOpt(v) match {
      case Some(dataType) => dataType
      case None           => throw new SilkDataFormatException("Unrecognized bag data type")
    }

  /** @group Values */
  case object SKBAG_FIELD_SIPv4 extends BagDataType(0)

  /** @group Values */
  case object SKBAG_FIELD_DIPv4 extends BagDataType(1)

  /** @group Values */
  case object SKBAG_FIELD_SPORT extends BagDataType(2)

  /** @group Values */
  case object SKBAG_FIELD_DPORT extends BagDataType(3)

  /** @group Values */
  case object SKBAG_FIELD_PROTO extends BagDataType(4)

  /** @group Values */
  case object SKBAG_FIELD_PACKETS extends BagDataType(5)

  /** @group Values */
  case object SKBAG_FIELD_BYTES extends BagDataType(6)

  /** @group Values */
  case object SKBAG_FIELD_FLAGS extends BagDataType(7)

  /** @group Values */
  case object SKBAG_FIELD_STARTTIME extends BagDataType(8)

  /** @group Values */
  case object SKBAG_FIELD_ELAPSED extends BagDataType(9)

  /** @group Values */
  case object SKBAG_FIELD_ENDTIME extends BagDataType(10)

  /** @group Values */
  case object SKBAG_FIELD_SID extends BagDataType(11)

  /** @group Values */
  case object SKBAG_FIELD_INPUT extends BagDataType(12)

  /** @group Values */
  case object SKBAG_FIELD_OUTPUT extends BagDataType(13)

  /** @group Values */
  case object SKBAG_FIELD_NHIPv4 extends BagDataType(14)

  /** @group Values */
  case object SKBAG_FIELD_INIT_FLAGS extends BagDataType(15)

  /** @group Values */
  case object SKBAG_FIELD_REST_FLAGS extends BagDataType(16)

  /** @group Values */
  case object SKBAG_FIELD_TCP_STATE extends BagDataType(17)

  /** @group Values */
  case object SKBAG_FIELD_APPLICATION extends BagDataType(18)

  /** @group Values */
  case object SKBAG_FIELD_FTYPE_CLASS extends BagDataType(19)

  /** @group Values */
  case object SKBAG_FIELD_FTYPE_TYPE extends BagDataType(20)

  /** @group Values */
  case object SKBAG_FIELD_ICMP_TYPE_CODE extends BagDataType(24)

  /** @group Values */
  case object SKBAG_FIELD_SIPv6 extends BagDataType(25)

  /** @group Values */
  case object SKBAG_FIELD_DIPv6 extends BagDataType(26)

  /** @group Values */
  case object SKBAG_FIELD_NHIPv6 extends BagDataType(27)

  /** @group Values */
  case object SKBAG_FIELD_RECORDS extends BagDataType(28)

  /** @group Values */
  case object SKBAG_FIELD_SUM_PACKETS extends BagDataType(29)

  /** @group Values */
  case object SKBAG_FIELD_SUM_BYTES extends BagDataType(30)

  /** @group Values */
  case object SKBAG_FIELD_SUM_ELAPSED extends BagDataType(31)

  /** @group Values */
  case object SKBAG_FIELD_ANY_IPv4 extends BagDataType(32)

  /** @group Values */
  case object SKBAG_FIELD_ANY_IPv6 extends BagDataType(33)

  /** @group Values */
  case object SKBAG_FIELD_ANY_PORT extends BagDataType(34)

  /** @group Values */
  case object SKBAG_FIELD_ANY_SNMP extends BagDataType(35)

  /** @group Values */
  case object SKBAG_FIELD_ANY_TIME extends BagDataType(36)

  /** @group Values */
  case object SKBAG_FIELD_SIP_COUNTRY extends BagDataType(37)

  /** @group Values */
  case object SKBAG_FIELD_DIP_COUNTRY extends BagDataType(38)

  /** @group Values */
  case object SKBAG_FIELD_ANY_COUNTRY extends BagDataType(39)

  /** @group Values */
  case object SKBAG_FIELD_SIP_PMAP extends BagDataType(40)

  /** @group Values */
  case object SKBAG_FIELD_DIP_PMAP extends BagDataType(41)

  /** @group Values */
  case object SKBAG_FIELD_ANY_IP_PMAP extends BagDataType(42)

  /** @group Values */
  case object SKBAG_FIELD_SPORT_PMAP extends BagDataType(43)

  /** @group Values */
  case object SKBAG_FIELD_DPORT_PMAP extends BagDataType(44)

  /** @group Values */
  case object SKBAG_FIELD_ANY_PORT_PMAP extends BagDataType(45)

  /** @group Values */
  case object SKBAG_FIELD_CUSTOM extends BagDataType(255)

}

// @LICENSE_FOOTER@
//
// Mothra 1.7
//
// Copyright 2025 Carnegie Mellon University.
//
// NO WARRANTY. THIS CARNEGIE MELLON UNIVERSITY AND SOFTWARE ENGINEERING INSTITUTE MATERIAL IS
// FURNISHED ON AN "AS-IS" BASIS. CARNEGIE MELLON UNIVERSITY MAKES NO WARRANTIES OF ANY KIND,
// EITHER EXPRESSED OR IMPLIED, AS TO ANY MATTER INCLUDING, BUT NOT LIMITED TO, WARRANTY OF FITNESS
// FOR PURPOSE OR MERCHANTABILITY, EXCLUSIVITY, OR RESULTS OBTAINED FROM USE OF THE MATERIAL.
// CARNEGIE MELLON UNIVERSITY DOES NOT MAKE ANY WARRANTY OF ANY KIND WITH RESPECT TO FREEDOM FROM
// PATENT, TRADEMARK, OR COPYRIGHT INFRINGEMENT.
//
// Licensed under a GNU GPL 2.0-style license, please see LICENSE.txt or contac
// permission@sei.cmu.edu for full terms.
//
// [DISTRIBUTION STATEMENT A] This material has been approved for public release and unlimited
// distribution.  Please see Copyright notice for non-US Government use and distribution.
//
// This Software includes and/or makes use of Third-Party Software each subject to its own license.
//
// DM24-1649
