// Copyright 2025 by Carnegie Mellon University
// See license information in LICENSE.txt

package org.cert.netsa.io.silk

import java.lang.{Byte => JByte}

import enumeratum.values.{ByteEnum, ByteEnumEntry}

/** A SiLK compression method. May be converted to and from [[scala.Byte Byte]] values.
  *
  * See [[CompressionMethod$ the companion object]] for more details.
  *
  * @param value The byte value representing this compression method.
  */
sealed abstract class CompressionMethod(val value: Byte) extends ByteEnumEntry {
  def toByte: Byte = value
  def toInt: Int = JByte.toUnsignedInt(value)
}

case object CompressionMethod extends ByteEnum[CompressionMethod] {

  /** Collection of all known valid compression methods.
    * @group Members
    */
  lazy val values = findValues
  // Note: lazy to avoid cyclic initialization problem

  /** Converts a byte value to a compression method.
    * @return The compression method represented by this value.
    * @throws SilkDataFormatException if the byte represents no known compression method.
    * @group Members
    */
  def apply(v: Byte): CompressionMethod =
    withValueOpt(v) match {
      case Some(method) => method
      case None         => throw new SilkDataFormatException(s"Unrecognized compression method $v")
    }

  /** SiLK compression method for no compression.
    * @group Values
    */
  case object NONE extends CompressionMethod(0)

  /** SiLK compression method for zlib compression.
    * @group Values
    */
  case object ZLIB extends CompressionMethod(1)

  /** SiLK compression method for lzo compression.
    * @group Values
    */
  case object LZO1X extends CompressionMethod(2)

  /** SiLK compression method for snappy compression.
    * @group Values
    */
  case object SNAPPY extends CompressionMethod(3)
}

// @LICENSE_FOOTER@
//
// Mothra 1.7
//
// Copyright 2025 Carnegie Mellon University.
//
// NO WARRANTY. THIS CARNEGIE MELLON UNIVERSITY AND SOFTWARE ENGINEERING INSTITUTE MATERIAL IS
// FURNISHED ON AN "AS-IS" BASIS. CARNEGIE MELLON UNIVERSITY MAKES NO WARRANTIES OF ANY KIND,
// EITHER EXPRESSED OR IMPLIED, AS TO ANY MATTER INCLUDING, BUT NOT LIMITED TO, WARRANTY OF FITNESS
// FOR PURPOSE OR MERCHANTABILITY, EXCLUSIVITY, OR RESULTS OBTAINED FROM USE OF THE MATERIAL.
// CARNEGIE MELLON UNIVERSITY DOES NOT MAKE ANY WARRANTY OF ANY KIND WITH RESPECT TO FREEDOM FROM
// PATENT, TRADEMARK, OR COPYRIGHT INFRINGEMENT.
//
// Licensed under a GNU GPL 2.0-style license, please see LICENSE.txt or contac
// permission@sei.cmu.edu for full terms.
//
// [DISTRIBUTION STATEMENT A] This material has been approved for public release and unlimited
// distribution.  Please see Copyright notice for non-US Government use and distribution.
//
// This Software includes and/or makes use of Third-Party Software each subject to its own license.
//
// DM24-1649
