// Copyright 2025 by Carnegie Mellon University
// See license information in LICENSE.txt

package org.cert.netsa.io.silk

import java.lang.{Byte => JByte}

import enumeratum.values.{ByteEnum, ByteEnumEntry}
import io.{FlowDecoder, FlowEncoder}
import io.flow_formats

/** A SiLK file format. May be converted to and from [[scala.Byte Byte]] values. Equipped with a
  * partial function mapping [[scala.Short Short]] file versions to [[io.FlowDecoder FlowDecoder]]s.
  *
  * See [[FileFormat$ the companion object]] for more details.
  *
  * @param value The byte value representing this file format.
  */
sealed abstract class FileFormat(
  val value: Byte,
  private val decoders: PartialFunction[Short, FlowDecoder] = PartialFunction.empty,
  private val encoders: PartialFunction[Short, FlowEncoder] = PartialFunction.empty
) extends ByteEnumEntry {
  def toByte: Byte = value
  def toInt: Int = JByte.toUnsignedInt(value)
}

object FileFormat extends ByteEnum[FileFormat] {

  /** Collection of all known valid SiLK file formats.
    * @group Members
    */
  lazy val values = findValues
  // Note: lazy to avoid cyclic initialization problem

  /** Converts a byte value to a file format.
    * @return The file format represented by this value.
    * @throws SilkDataFormatException if the byte represents no known file format.
    * @group Members
    */
  def apply(v: Byte): FileFormat =
    withValueOpt(v) match {
      case Some(format) => format
      case None         => throw new SilkDataFormatException("Unrecognized file format")
    }

  /** Optionally returns the decoder for a given file format and record version, if such a decoder
    * exists.
    *
    * @return `Some(decoder)` if an appropriate decoder exists, `None` otherwise.
    * @group Members
    */
  def decoder(fileFormat: FileFormat, recordVersion: Short): Option[FlowDecoder] =
    fileFormat.decoders.lift(recordVersion)

  /** Optionally returns the encoder for a given file format and record version, if such an encoder
    * exists.
    *
    * @return `Some(encoder)` if an appropriate encoder exists, `None` otherwise.
    * @group Members
    */
  def encoder(fileFormat: FileFormat, recordVersion: Short): Option[FlowEncoder] =
    fileFormat.encoders.lift(recordVersion)

  /** @group Values */
  case object FT_TCPDUMP extends FileFormat(0x00)

  /** @group Values */
  case object FT_GRAPH extends FileFormat(0x01)

  /** @group Values */
  case object FT_ADDRESSES extends FileFormat(0x02)

  /** @group Values */
  case object FT_PORTMAP extends FileFormat(0x03)

  /** @group Values */
  case object FT_SERVICEMAP extends FileFormat(0x04)

  /** @group Values */
  case object FT_NIDSMAP extends FileFormat(0x05)

  /** @group Values */
  case object FT_EXPERIMENT1 extends FileFormat(0x06)

  /** @group Values */
  case object FT_EXPERIMENT2 extends FileFormat(0x07)

  /** @group Values */
  case object FT_TEMPFILE extends FileFormat(0x08)

  /** @group Values */
  case object FT_RESERVED_09 extends FileFormat(0x09)

  /** @group Values */
  case object FT_IPFIX extends FileFormat(0x0a)

  /** @group Values */
  case object FT_RWIPV6
      extends FileFormat(
        0x0b,
        decoders = {
          case 1 => flow_formats.FT_RWIPV6_v1
          case 2 => flow_formats.FT_RWIPV6_v2
          case 3 => flow_formats.FT_RWIPV6_v3
        }
      )

  /** @group Values */
  case object FT_RWIPV6ROUTING
      extends FileFormat(
        0x0c,
        decoders = {
          case 1 => flow_formats.FT_RWIPV6ROUTING_v1
          case 2 => flow_formats.FT_RWIPV6ROUTING_v1 // TOS hack
          case 3 => flow_formats.FT_RWIPV6ROUTING_v3
          case 4 => flow_formats.FT_RWIPV6ROUTING_v4
        },
        encoders = {
          case 1 => flow_formats.FT_RWIPV6ROUTING_v1
          case 4 => flow_formats.FT_RWIPV6ROUTING_v4
        }
      )

  /** @group Values */
  case object FT_RWAUGSNMPOUT
      extends FileFormat(
        0x0d,
        decoders = {
          // V1 and V2 differ only in the padding of the header
          // V2 and V3 differ only in that V3 supports compression on
          //   read and write; V2 supports compression only on read
          case 1 => flow_formats.FT_RWAUGSNMPOUT_v1
          case 2 => flow_formats.FT_RWAUGSNMPOUT_v1
          case 3 => flow_formats.FT_RWAUGSNMPOUT_v1
          case 4 => flow_formats.FT_RWAUGSNMPOUT_v4
          case 5 => flow_formats.FT_RWAUGSNMPOUT_v5
        }
      )

  /** @group Values */
  case object FT_RWAUGROUTING
      extends FileFormat(
        0x0e,
        decoders = {
          // V1 and V2 differ only in the padding of the header
          // V2 and V3 differ only in that V3 supports compression on
          //   read and write; V2 supports compression only on read
          case 1 => flow_formats.FT_RWAUGROUTING_v1
          case 2 => flow_formats.FT_RWAUGROUTING_v1
          case 3 => flow_formats.FT_RWAUGROUTING_v1
          case 4 => flow_formats.FT_RWAUGROUTING_v4
          case 5 => flow_formats.FT_RWAUGROUTING_v5
          case 6 => flow_formats.FT_RWAUGROUTING_v6
        }
      )

  /** @group Values */
  case object FT_RESERVED_0F extends FileFormat(0x0f)

  /** @group Values */
  case object FT_RWROUTED
      extends FileFormat(
        0x10,
        decoders = {
          // V1 and V2 differ only in the padding of the header
          case 1 => flow_formats.FT_RWROUTED_v1
          case 2 => flow_formats.FT_RWROUTED_v1
          // V3 and V4 differ only in that V4 supports compression on
          //   read and write; V3 supports compression only on read
          case 3 => flow_formats.FT_RWROUTED_v3
          case 4 => flow_formats.FT_RWROUTED_v3
          case 5 => flow_formats.FT_RWROUTED_v5
        }
      )

  /** @group Values */
  case object FT_RWNOTROUTED
      extends FileFormat(
        0x11,
        decoders = {
          // V1 and V2 differ only in the padding of the header
          case 1 => flow_formats.FT_RWNOTROUTED_v1
          case 2 => flow_formats.FT_RWNOTROUTED_v1
          // V3 and V4 differ only in that V4 supports compression on
          //   read and write; V3 supports compression only on read
          case 3 => flow_formats.FT_RWNOTROUTED_v3
          case 4 => flow_formats.FT_RWNOTROUTED_v3
          case 5 => flow_formats.FT_RWNOTROUTED_v5
        }
      )

  /** @group Values */
  case object FT_RWSPLIT
      extends FileFormat(
        0x12,
        decoders = {
          // V1 and V2 differ only in the padding of the header
          case 1 => flow_formats.FT_RWSPLIT_v1
          case 2 => flow_formats.FT_RWSPLIT_v1
          // V3 and V4 differ only in that V4 supports compression on
          //   read and write; V3 supports compression only on read
          case 3 => flow_formats.FT_RWSPLIT_v3
          case 4 => flow_formats.FT_RWSPLIT_v3
          case 5 => flow_formats.FT_RWSPLIT_v5
        }
      )

  /** @group Values */
  case object FT_RWFILTER
      extends FileFormat(
        0x13,
        decoders = {
          case 1 => flow_formats.FT_RWFILTER_v1
          case 2 => flow_formats.FT_RWFILTER_v2
          case 3 => flow_formats.FT_RWFILTER_v3
          // V4 and V5 differ only in that V5 supports compression on
          //   read and write; V4 supports compression only on read
          case 4 => flow_formats.FT_RWFILTER_v4
          case 5 => flow_formats.FT_RWFILTER_v4
        }
      )

  /** @group Values */
  case object FT_RWAUGMENTED
      extends FileFormat(
        0x14,
        decoders = {
          // V1 and V2 differ only in the padding of the header
          // V2 and V3 differ only in that V3 supports compression on
          //   read and write; V2 supports compression only on read
          case 1 => flow_formats.FT_RWAUGMENTED_v1
          case 2 => flow_formats.FT_RWAUGMENTED_v1
          case 3 => flow_formats.FT_RWAUGMENTED_v1
          case 4 => flow_formats.FT_RWAUGMENTED_v4
          case 5 => flow_formats.FT_RWAUGMENTED_v5
          case 6 => flow_formats.FT_RWAUGMENTED_v6
        }
      )

  /** @group Values */
  case object FT_RWAUGWEB
      extends FileFormat(
        0x15,
        decoders = {
          // V1 and V2 differ only in the padding of the header
          // V2 and V3 differ only in that V3 supports compression on
          //   read and write; V2 supports compression only on read
          case 1 => flow_formats.FT_RWAUGWEB_v1
          case 2 => flow_formats.FT_RWAUGWEB_v1
          case 3 => flow_formats.FT_RWAUGWEB_v1
          case 4 => flow_formats.FT_RWAUGWEB_v4
          case 5 => flow_formats.FT_RWAUGWEB_v5
          case 6 => flow_formats.FT_RWAUGWEB_v6
        }
      )

  /** @group Values */
  case object FT_RWGENERIC
      extends FileFormat(
        0x16,
        decoders = {
          case 0 => flow_formats.FT_RWGENERIC_v0
          case 1 => flow_formats.FT_RWGENERIC_v1
          case 2 => flow_formats.FT_RWGENERIC_v2
          case 3 => flow_formats.FT_RWGENERIC_v3
          case 4 => flow_formats.FT_RWGENERIC_v3
          case 5 => flow_formats.FT_RWGENERIC_v5
          case 6 => flow_formats.FT_RWGENERIC_v6
        }
      )

  /** @group Values */
  case object FT_RESERVED_17 extends FileFormat(0x17)

  /** @group Values */
  case object FT_RWDAILY extends FileFormat(0x18)

  /** @group Values */
  case object FT_RWSCAN extends FileFormat(0x19)

  /** @group Values */
  case object FT_RWACL extends FileFormat(0x1a)

  /** @group Values */
  case object FT_RWCOUNT extends FileFormat(0x1b)

  /** @group Values */
  case object FT_FLOWCAP
      extends FileFormat(
        0x1c,
        decoders = {
          case 2 => flow_formats.FT_FLOWCAP_v2
          case 3 => flow_formats.FT_FLOWCAP_v3
          case 4 => flow_formats.FT_FLOWCAP_v4
          case 5 => flow_formats.FT_FLOWCAP_v5
          case 6 => flow_formats.FT_FLOWCAP_v5 // TOS hack
        }
      )

  /** @group Values */
  case object FT_IPSET extends FileFormat(0x1d)

  /** @group Values */
  case object FT_TAGTREE extends FileFormat(0x1e)

  /** @group Values */
  case object FT_RWWWW
      extends FileFormat(
        0x1f,
        decoders = {
          // V1 and V2 differ only in the padding of the header
          case 1 => flow_formats.FT_RWWWW_v1
          case 2 => flow_formats.FT_RWWWW_v1
          // V3 and V4 differ only in that V4 supports compression on
          //   read and write; V3 supports compression only on read
          case 3 => flow_formats.FT_RWWWW_v3
          case 4 => flow_formats.FT_RWWWW_v3
          case 5 => flow_formats.FT_RWWWW_v5
        }
      )

  /** @group Values */
  case object FT_SHUFFLE extends FileFormat(0x20)

  /** @group Values */
  case object FT_RWBAG extends FileFormat(0x21)

  /** @group Values */
  case object FT_BLOOM extends FileFormat(0x22)

  /** @group Values */
  case object FT_RWPRINTSTATS extends FileFormat(0x23)

  /** @group Values */
  case object FT_PDUFLOWCAP extends FileFormat(0x24)

  /** @group Values */
  case object FT_PREFIXMAP extends FileFormat(0x25)

}

// @LICENSE_FOOTER@
//
// Mothra 1.7
//
// Copyright 2025 Carnegie Mellon University.
//
// NO WARRANTY. THIS CARNEGIE MELLON UNIVERSITY AND SOFTWARE ENGINEERING INSTITUTE MATERIAL IS
// FURNISHED ON AN "AS-IS" BASIS. CARNEGIE MELLON UNIVERSITY MAKES NO WARRANTIES OF ANY KIND,
// EITHER EXPRESSED OR IMPLIED, AS TO ANY MATTER INCLUDING, BUT NOT LIMITED TO, WARRANTY OF FITNESS
// FOR PURPOSE OR MERCHANTABILITY, EXCLUSIVITY, OR RESULTS OBTAINED FROM USE OF THE MATERIAL.
// CARNEGIE MELLON UNIVERSITY DOES NOT MAKE ANY WARRANTY OF ANY KIND WITH RESPECT TO FREEDOM FROM
// PATENT, TRADEMARK, OR COPYRIGHT INFRINGEMENT.
//
// Licensed under a GNU GPL 2.0-style license, please see LICENSE.txt or contac
// permission@sei.cmu.edu for full terms.
//
// [DISTRIBUTION STATEMENT A] This material has been approved for public release and unlimited
// distribution.  Please see Copyright notice for non-US Government use and distribution.
//
// This Software includes and/or makes use of Third-Party Software each subject to its own license.
//
// DM24-1649
