// Copyright 2025 by Carnegie Mellon University
// See license information in LICENSE.txt

package org.cert.netsa.io.silk

/** One of a variety of SiLK extended header entries.
  *
  * See [[HeaderEntry$ the companion object]] for more details.
  */
sealed abstract class HeaderEntry protected ()

object HeaderEntry {

  /** Internally used to represent the end of headers when reading in header entries. */
  private[silk] case object EndOfHeaders extends HeaderEntry

  /** Header entry used for data files generated by rwflowpack. It specifies the following fields
    * that are simplified or the same for all entries in a packed file. (Times are offsets from
    * startTime, flowtypeId and sensorId are the same for all records.)
    *
    * @param startTime Base start time, in milliseconds since the UNIX epoch. Times in this file are
    *   expressed as a delta from this base start time.
    * @param flowtypeId SiLK flow type for all flows in this file.
    * @param sensorId SiLK sensor ID for all flows in this file.
    * @group Values
    */
  case class PackedFile(val startTime: Long, val flowtypeId: FlowType, val sensorId: Sensor)
      extends HeaderEntry

  /** Header entry used to store the command line history, with one instance per command invoked to
    * produce the file.
    *
    * @param commandLine UNIX command line used to generate this file.
    * @group Values
    */
  case class Invocation(val commandLine: String) extends HeaderEntry

  /** Header entry used to store a generic comment or annotation about the file.
    *
    * @param annotation Content of text note.
    * @group Values
    */
  case class Annotation(val annotation: String) extends HeaderEntry

  /** Header entry used to store the textual name of the probe where flow data was originally
    * collected.
    *
    * @param probeName The name of the probe from which data was collected.
    * @group Values
    */
  case class ProbeName(val probeName: String) extends HeaderEntry

  /** Header entry used to store information particular to prefix maps (pmaps). Not used for flow
    * data.
    *
    * @param version The version of this header entry (always 1).
    * @param mapName Textual human-readable name of this prefix map.
    * @group Values
    */
  case class PrefixMap(val version: Int, val mapName: String) extends HeaderEntry

  /** Header entry used to store information particular to binary bag files. Not used for flow data.
    *
    * @param keyType
    * @param keyLength
    * @param counterType
    * @param counterLength
    * @group Values
    */
  case class Bag(
    val keyType: Short,
    val keyLength: Short,
    val counterType: Short,
    val counterLength: Short
  ) extends HeaderEntry

  /** Header entry used to store information particular to binary IP sets. Not used for flow data.
    *
    * @param childNode
    * @param leafCount
    * @param leafSize
    * @param nodeCount
    * @param nodeSize
    * @param rootIndex
    * @group Values
    */
  case class IPSet(
    val childNode: Int,
    val leafCount: Int,
    val leafSize: Int,
    val nodeCount: Int,
    val nodeSize: Int,
    val rootIndex: Int
  ) extends HeaderEntry

  /** Header entry used for headers that are unrecognized.
    *
    * @param id Numeric header entry ID
    * @param data Binary header entry data
    * @group Values
    */
  case class Unknown(val id: Int, val data: Array[Byte]) extends HeaderEntry

}

// @LICENSE_FOOTER@
//
// Mothra 1.7
//
// Copyright 2025 Carnegie Mellon University.
//
// NO WARRANTY. THIS CARNEGIE MELLON UNIVERSITY AND SOFTWARE ENGINEERING INSTITUTE MATERIAL IS
// FURNISHED ON AN "AS-IS" BASIS. CARNEGIE MELLON UNIVERSITY MAKES NO WARRANTIES OF ANY KIND,
// EITHER EXPRESSED OR IMPLIED, AS TO ANY MATTER INCLUDING, BUT NOT LIMITED TO, WARRANTY OF FITNESS
// FOR PURPOSE OR MERCHANTABILITY, EXCLUSIVITY, OR RESULTS OBTAINED FROM USE OF THE MATERIAL.
// CARNEGIE MELLON UNIVERSITY DOES NOT MAKE ANY WARRANTY OF ANY KIND WITH RESPECT TO FREEDOM FROM
// PATENT, TRADEMARK, OR COPYRIGHT INFRINGEMENT.
//
// Licensed under a GNU GPL 2.0-style license, please see LICENSE.txt or contac
// permission@sei.cmu.edu for full terms.
//
// [DISTRIBUTION STATEMENT A] This material has been approved for public release and unlimited
// distribution.  Please see Copyright notice for non-US Government use and distribution.
//
// This Software includes and/or makes use of Third-Party Software each subject to its own license.
//
// DM24-1649
