// Copyright 2025 by Carnegie Mellon University
// See license information in LICENSE.txt

package org.cert.netsa.io.silk

import org.cert.netsa.data.net.{Port, Protocol}

/** A type representing a range of ([[org.cert.netsa.data.net.Protocol]],
  * [[org.cert.netsa.data.net.Port]]) Pairs.
  *
  * This is the type of the key when using an [[scala.collection.Iterator Iterator]] over the
  * contents of a PrefixMap file that contains (Protocol, Port) Pairs.
  *
  * @see [[PrefixMapReader]]
  */
final case class PrefixMapProtocolPortPair(
  val startProtocol: Protocol,
  val startPort: Port,
  val endProtocol: Protocol,
  val endPort: Port
) {
  require(
    startProtocol.toInt <= endProtocol.toInt,
    s"start protocol (${startProtocol.toInt}) > end (${endProtocol.toInt})"
  )

  require(
    startProtocol != endProtocol || startPort.toInt <= endPort.toInt,
    s"start port (${startPort.toInt}) > end (${endPort.toInt})"
  )
}

/** The PrefixMapProtocolPortPair object provides support for creating a
  * [[PrefixMapProtocolPortPair]].
  */
object PrefixMapProtocolPortPair {

  /** Creates a PrefixMapProtocolPortPair from four integers without requiring the caller to create
    * the appropriate type from each.
    */
  def apply(loProto: Int, loPort: Int, hiProto: Int, hiPort: Int): PrefixMapProtocolPortPair =
    new PrefixMapProtocolPortPair(Protocol(loProto), Port(loPort), Protocol(hiProto), Port(hiPort))
}

// @LICENSE_FOOTER@
//
// Mothra 1.7
//
// Copyright 2025 Carnegie Mellon University.
//
// NO WARRANTY. THIS CARNEGIE MELLON UNIVERSITY AND SOFTWARE ENGINEERING INSTITUTE MATERIAL IS
// FURNISHED ON AN "AS-IS" BASIS. CARNEGIE MELLON UNIVERSITY MAKES NO WARRANTIES OF ANY KIND,
// EITHER EXPRESSED OR IMPLIED, AS TO ANY MATTER INCLUDING, BUT NOT LIMITED TO, WARRANTY OF FITNESS
// FOR PURPOSE OR MERCHANTABILITY, EXCLUSIVITY, OR RESULTS OBTAINED FROM USE OF THE MATERIAL.
// CARNEGIE MELLON UNIVERSITY DOES NOT MAKE ANY WARRANTY OF ANY KIND WITH RESPECT TO FREEDOM FROM
// PATENT, TRADEMARK, OR COPYRIGHT INFRINGEMENT.
//
// Licensed under a GNU GPL 2.0-style license, please see LICENSE.txt or contac
// permission@sei.cmu.edu for full terms.
//
// [DISTRIBUTION STATEMENT A] This material has been approved for public release and unlimited
// distribution.  Please see Copyright notice for non-US Government use and distribution.
//
// This Software includes and/or makes use of Third-Party Software each subject to its own license.
//
// DM24-1649
