// Copyright 2025 by Carnegie Mellon University
// See license information in LICENSE.txt

package org.cert.netsa.io.silk

import java.time.{Duration, Instant}

import org.cert.netsa.data.net.{
  ICMPCode, ICMPType, IPAddress, Port, Protocol, SNMPInterface, TCPFlags
}

// Could make some of these Options, based on not all record types
// having them, and also based on not all records having them
// (i.e. icmpType and icmpCode are meaningful for ICMP but not TCP,
// flags &c only useful for TCP, etc.

/** A SiLK flow record.
  *
  * Note that in addition to the fields of the case class, some derived fields are also provided.
  * (See below.)
  *
  * @param startTime The instant that the first packet in this flow was observed.
  * @param endTime The instant that the last packet in this flow was observed.
  * @param sPort The source port of this flow, or zero if this flow is neither a TCP nor a UDP flow.
  * @param dPort The destination port of this flow, or zero if this flow is neither a TCP nor a UDP
  *   flow.
  * @param protocol The IP protocol of this flow.
  * @param flowType The SiLK flow type (class and type) of this flow, or `FlowType(0)` if unknown.
  * @param sensor The SiLK sensor that observed this flow, or `Sensor(0)` if unknown.
  * @param flags The union of all TCP flags observed in this flow, or `TCPFlags(0)` if this flow is
  *   not a TCP flow.
  * @param initFlags The flags observed in the initial packet of this TCP flow, or `TCPFlags(0)` if
  *   this flow is not a TCP flow or if extended flags are not available.
  * @param restFlags The union of all TCP flags observed after the initial packet of this flow, or
  *   `TCPFlags(0)` if this flow is not a TCP flow or if extended flags are not available.
  * @param tcpState Flags relating to the observed status of this flow, including whether extended
  *   TCP flags are available. See [[TCPState]] for more details.
  * @param application The detected application of this flow, expressed as the common port number
  *   for that application, or `Port(0)` if no application was detected.
  * @param memo A [[scala.Short Short]] value stored as a memo on this flow, or zero if no such memo
  *   has been set.
  * @param input The input SNMP routing interface for this flow, or `SNMPInterface(0)` if routing
  *   information is not available.
  * @param output The output SNMP routing interface for this flow, or `SNMPInterface(0)` if routing
  *   information is not available.
  * @param packets The number of IP packets observed in this flow.
  * @param bytes The number of bytes in packets observed in this flow.
  * @param sIP The source IP address of packets in this flow.
  * @param dIP The destination IP address of packets in this flow.
  * @param nhIP The next-hop IP address of packets in this flow, or `IPAddress("0.0.0.0")` or
  *   `IPAddress("::")` if routing information is not available.
  */
case class RWRec(
  val startTime: Instant,
  val endTime: Instant,
  val sPort: Port,
  val dPort: Port,
  val protocol: Protocol,
  val flowType: FlowType,
  val sensor: Sensor,
  val flags: TCPFlags,
  val initFlags: TCPFlags,
  val restFlags: TCPFlags,
  val tcpState: TCPState,
  val application: Port,
  val memo: Short,
  val input: SNMPInterface,
  val output: SNMPInterface,
  val packets: Long,
  val bytes: Long,
  val sIP: IPAddress,
  val dIP: IPAddress,
  val nhIP: IPAddress
) {

  /** The duration between the instants the first and last packets in this flow were observed.
    * @group Derived Members
    */
  def elapsed: Duration = Duration.between(startTime, endTime)

  /** The type of this ICMP flow, or garbage if this is a TCP or UDP flow, or `ICMPType(0)` if this
    * is not an ICMP, TCP, or UDP flow.
    * @group Derived Members
    */
  def icmpType: ICMPType = ICMPType((dPort.toShort >> 8).toByte)

  /** The code of this ICMP flow, or garbage if this is a TCP or UDP flow, or `ICMPType(0)` if this
    * is not an ICMP, TCP, or UDP flow.
    * @group Derived Members
    */
  def icmpCode: ICMPCode = ICMPCode(dPort.toShort.toByte)

  /** True if this flow's addresses are IPv6 addresses.
    * @group Derived Members
    */
  def isIPv6: Boolean = tcpState.isIPv6
}

object RWRec {
  def apply(
    startTime: Instant,
    elapsed: Duration,
    sPort: Port,
    dPort: Port,
    protocol: Protocol,
    flowType: FlowType,
    sensor: Sensor,
    flags: TCPFlags,
    initFlags: TCPFlags,
    restFlags: TCPFlags,
    tcpState: TCPState,
    application: Port,
    memo: Short,
    input: SNMPInterface,
    output: SNMPInterface,
    packets: Long,
    bytes: Long,
    sIP: IPAddress,
    dIP: IPAddress,
    nhIP: IPAddress
  ): RWRec =
    RWRec(
      startTime, startTime.plus(elapsed), sPort, dPort, protocol, flowType, sensor, flags,
      initFlags, restFlags, tcpState, application, memo, input, output, packets, bytes, sIP, dIP,
      nhIP
    )
}

// @LICENSE_FOOTER@
//
// Mothra 1.7
//
// Copyright 2025 Carnegie Mellon University.
//
// NO WARRANTY. THIS CARNEGIE MELLON UNIVERSITY AND SOFTWARE ENGINEERING INSTITUTE MATERIAL IS
// FURNISHED ON AN "AS-IS" BASIS. CARNEGIE MELLON UNIVERSITY MAKES NO WARRANTIES OF ANY KIND,
// EITHER EXPRESSED OR IMPLIED, AS TO ANY MATTER INCLUDING, BUT NOT LIMITED TO, WARRANTY OF FITNESS
// FOR PURPOSE OR MERCHANTABILITY, EXCLUSIVITY, OR RESULTS OBTAINED FROM USE OF THE MATERIAL.
// CARNEGIE MELLON UNIVERSITY DOES NOT MAKE ANY WARRANTY OF ANY KIND WITH RESPECT TO FREEDOM FROM
// PATENT, TRADEMARK, OR COPYRIGHT INFRINGEMENT.
//
// Licensed under a GNU GPL 2.0-style license, please see LICENSE.txt or contac
// permission@sei.cmu.edu for full terms.
//
// [DISTRIBUTION STATEMENT A] This material has been approved for public release and unlimited
// distribution.  Please see Copyright notice for non-US Government use and distribution.
//
// This Software includes and/or makes use of Third-Party Software each subject to its own license.
//
// DM24-1649
