// Copyright 2025 by Carnegie Mellon University
// See license information in LICENSE.txt

package org.cert.netsa.io.silk

import java.io.{EOFException, InputStream}
import java.nio.{ByteBuffer, ByteOrder}
import java.time.temporal.ChronoUnit

import io.{
  BufferReader, FlowDecoder, LzoInputStreamBuffer, RawInputStreamBuffer, SnappyInputStreamBuffer,
  ZlibInputStreamBuffer
}

/** A reader of SiLK flow records. This is usable as an [[scala.Iterator Iterator]] over [[RWRec]]
  * records.
  *
  * See [[RWRecReader$ the companion object]] for more details.
  */
class RWRecReader private (
  private val reader: BufferReader,
  private val decoder: FlowDecoder,
  val header: Header
) extends Iterator[RWRec] {

  /** Current buffer of bytes being processed. */
  private var buffer = ByteBuffer.allocate(0)

  /** Length of current buffer in bytes. */
  private var bufferLength = 0

  /** Current offset into current buffer. */
  private var offset = 0

  /** True if reading of the input has completed, and no more records should be read or produced. */
  private var done = false

  /** The length in bytes of records being processed by this reader. */
  private val recordLength = decoder.recordLength

  /** Read a new buffer into memory if needed. Does no work if the current buffer still has unread
    * records, or if the reader is done. Sets done to true if the end of file is reached.
    */
  private def fillBuffer(): Unit = {
    if (!done) {
      if (offset + recordLength > bufferLength) {
        try {
          val (newBuffer, newLength) = reader.getNextBuffer()
          buffer = ByteBuffer.wrap(newBuffer)
          buffer.order(
            if (header.isBigEndian) ByteOrder.BIG_ENDIAN
            else ByteOrder.LITTLE_ENDIAN
          )
          bufferLength = newLength
          offset = 0
        } catch {
          case _: EOFException => {
            reader.close()
            buffer = ByteBuffer.allocate(0)
            done = true
          }
        }
      }
    }
  }

  /** The precision of times provided by this record source. */
  def timePrecision: ChronoUnit = decoder.timePrecision

  override def hasNext: Boolean = {
    fillBuffer()
    !done
  }

  override def next(): RWRec = {
    fillBuffer()
    if (done) {
      throw new NoSuchElementException("RWRecReader")
    }
    val rec = decoder.decode(buffer, offset, header)
    offset += recordLength
    rec
  }

}

object RWRecReader {

  /** Creates and returns a reader from the provided input stream. Does not support compressed data.
    *
    * @throws SilkDataFormatException if the input stream is malformed or uses compression.
    */
  def ofInputStream(s: InputStream): RWRecReader = {
    val header = Header.readFrom(s)
    val decoder = FileFormat.decoder(header.fileFormat, header.recordVersion) match {
      case Some(decoder) => decoder
      case None => throw new SilkDataFormatException("Unrecognized file format and version")
    }
    val bufferSize = (65536 / decoder.recordLength) * decoder.recordLength
    val bufferReader = header.compressionMethod match {
      case CompressionMethod.NONE   => RawInputStreamBuffer(s, bufferSize)
      case CompressionMethod.ZLIB   => ZlibInputStreamBuffer(s)
      case CompressionMethod.LZO1X  => LzoInputStreamBuffer(s)
      case CompressionMethod.SNAPPY => SnappyInputStreamBuffer(s)
      case _ => throw new SilkDataFormatException("Unsupported compression method (without Hadoop)")
    }

    new RWRecReader(bufferReader, decoder, header)
  }

}

// @LICENSE_FOOTER@
//
// Mothra 1.7
//
// Copyright 2025 Carnegie Mellon University.
//
// NO WARRANTY. THIS CARNEGIE MELLON UNIVERSITY AND SOFTWARE ENGINEERING INSTITUTE MATERIAL IS
// FURNISHED ON AN "AS-IS" BASIS. CARNEGIE MELLON UNIVERSITY MAKES NO WARRANTIES OF ANY KIND,
// EITHER EXPRESSED OR IMPLIED, AS TO ANY MATTER INCLUDING, BUT NOT LIMITED TO, WARRANTY OF FITNESS
// FOR PURPOSE OR MERCHANTABILITY, EXCLUSIVITY, OR RESULTS OBTAINED FROM USE OF THE MATERIAL.
// CARNEGIE MELLON UNIVERSITY DOES NOT MAKE ANY WARRANTY OF ANY KIND WITH RESPECT TO FREEDOM FROM
// PATENT, TRADEMARK, OR COPYRIGHT INFRINGEMENT.
//
// Licensed under a GNU GPL 2.0-style license, please see LICENSE.txt or contac
// permission@sei.cmu.edu for full terms.
//
// [DISTRIBUTION STATEMENT A] This material has been approved for public release and unlimited
// distribution.  Please see Copyright notice for non-US Government use and distribution.
//
// This Software includes and/or makes use of Third-Party Software each subject to its own license.
//
// DM24-1649
