// Copyright 2025 by Carnegie Mellon University
// See license information in LICENSE.txt

package org.cert.netsa.io.silk

import java.io.{DataOutputStream, OutputStream}
import java.nio.{ByteBuffer, ByteOrder}
import java.time.temporal.ChronoUnit

import io.{
  BufferWriter, FlowEncoder, LzoOutputStreamBuffer, RawOutputStreamBuffer, SnappyOutputStreamBuffer,
  ZlibOutputStreamBuffer
}

/** A writer of binary SiLK RWRec files that are readable by SiLK.
  *
  * @example
  *   This example reads the contents of "example.rw" and writes it to "copy.rw":
  *   {{{
  * val in = new java.io.FileInputStream("example.rw")
  * val out = new java.io.FileOutputStream("copy.rw")
  * val reader = RWRecReader.ofInputStream(in)
  * val writer = RWRecWriter.forOutputStream(out)
  * writer.append(reader)
  * writer.close()
  *   }}}
  *
  * @see [[RWRecWriter$ the companion object]] for more details
  */
class RWRecWriter private (
  val out: DataOutputStream,
  val compressionMethod: CompressionMethod,
  val fileFormat: FileFormat,
  val recordVersion: Short,
  val order: ByteOrder
) {

  /** Set to true once the file's header has been written */
  private var headerWritten = false

  private val encoder: FlowEncoder = FileFormat.encoder(fileFormat, recordVersion) match {
    case Some(encoder) => encoder
    case None => throw new SilkDataFormatException(
        s"Unsupported file format and version for output: $fileFormat v$recordVersion"
      )
  }

  // The header object for this writer
  private val header = new Header(
    if (order == ByteOrder.BIG_ENDIAN) Header.BigEndian
    else Header.LittleEndian,
    fileFormat,
    Header.FileVersion,
    compressionMethod,
    SilkVersion(0),
    encoder.recordLength,
    recordVersion,
    Vector(HeaderEntry.EndOfHeaders)
  )

  /** The size of the buffer to hold data prior to compressing. */
  private val bufferSize = (65536 / encoder.recordLength) * encoder.recordLength

  /** The buffer to hold data prior to compressing. The append() method fills this buffer. */
  private val buffer = ByteBuffer.allocate(bufferSize)
  buffer.order(order)

  /** The current offset into the output buffer. */
  private var offset = 0

  /** Object use to write/compress the output */
  private val writer: BufferWriter = compressionMethod match {
    case CompressionMethod.NONE   => RawOutputStreamBuffer(out)
    case CompressionMethod.ZLIB   => ZlibOutputStreamBuffer(out)
    case CompressionMethod.LZO1X  => LzoOutputStreamBuffer(out)
    case CompressionMethod.SNAPPY => SnappyOutputStreamBuffer(out)
    case _ => throw new SilkDataFormatException("Unrecognized compression method")
  }

  /** Writes the SiLK file header to the output stream. */
  private def writeHeader(): Unit = {
    header.writeTo(out)
    out.flush()
    headerWritten = true
  }

  /** Indicates whether the SiLK file header has been written to the output stream.
    *
    * @return `true` if header has been written.
    */
  def wasHeaderWritten: Boolean = headerWritten

  /** Iterates over the [[org.cert.netsa.io.silk.RWRec RWRecs]] and appends them to the destination
    * stream. Writes the SiLK file header to the output stream on the first call to this function.
    *
    * This function may be called multiple times.
    */
  def append(iter: Iterator[RWRec]): Unit = {
    if (!headerWritten) {
      writeHeader()
    }

    // process the records
    for (rwrec <- iter) {
      if (bufferSize - offset < encoder.recordLength) {
        writer.putBuffer(buffer.array, offset)
        offset = 0
      }
      encoder.encode(buffer, rwrec, offset, header)
      offset += encoder.recordLength
    }

    if (offset > 0) {
      writer.putBuffer(buffer.array, offset)
      offset = 0
    }
    out.flush()
  }

  /** Closes the output stream.
    *
    * Writes the SiLK file header to the output stream if it has not been written, writes any
    * buffered records, closes the output stream, and releases resources.
    */
  def close(): Unit = {
    if (!headerWritten) {
      writeHeader()
    }
    if (offset > 0) {
      writer.putBuffer(buffer.array, offset)
      offset = 0
    }
    writer.end()
    out.close()
  }

}

/** The RWRecWriter companion object provides support for creating an [[RWRecWriter]]. */
object RWRecWriter {

  /** Creates and returns a writer that writes [[org.cert.netsa.io.silk.RWRec RWRecs]] as a binary
    * SiLK RWRec stream to the output stream `s` using the default file format and version. The
    * default format preserves as much information as possible, but note that this may limit
    * compatibility with older versions of SiLK. For finer-grained control over the output format,
    * consider using [[forOutputStreamFormat]] or [[forOutputStreamPrecision]].
    *
    * If `compressionMethod` is provided, compresses output using that method, otherwise output is
    * not compressed.
    *
    * If `order` is provided, writes data in the specified byte order, otherwise uses native order.
    */
  def forOutputStream(
    s: OutputStream,
    compressionMethod: CompressionMethod = CompressionMethod.NONE,
    order: ByteOrder = ByteOrder.nativeOrder()
  ): RWRecWriter =
    forOutputStreamFormat(s, FileFormat.FT_RWIPV6ROUTING, 4, compressionMethod, order)

  /** Creates and returns a writer that writes [[org.cert.netsa.io.silk.RWRec RWRecs]] as a binary
    * SiLK RWRec stream to the output stream `s` using the requested file format and record version,
    * if available. @throws [[SilkDataFormatException]] if the format is not supported.
    *
    * If `compressionMethod` is provided, compresses output using that method, otherwise output is
    * not compressed.
    *
    * If `order` is provided, writes data in the specified byte order, otherwise uses native order.
    */
  def forOutputStreamFormat(
    s: OutputStream,
    fileFormat: FileFormat,
    recordVersion: Short,
    compressionMethod: CompressionMethod = CompressionMethod.NONE,
    order: ByteOrder = ByteOrder.nativeOrder()
  ): RWRecWriter = {
    val out = s match {
      case x: DataOutputStream => x
      case y: OutputStream     => new DataOutputStream(y)
    }

    new RWRecWriter(out, compressionMethod, fileFormat, recordVersion, order)
  }

  /** Creates and returns a writer that writes [[org.cert.netsa.io.silk.RWRec RWRecs]] as a binary
    * SiLK RWRec stream to the output stream `s` using the default file format and record version
    * for the requested time precision.
    *
    * If `compressionMethod` is provided, compresses output using that method, otherwise output is
    * not compressed.
    *
    * If `order` is provided, writes data in the specified byte order, otherwise uses native order.
    */
  def forOutputStreamPrecision(
    s: OutputStream,
    timePrecision: ChronoUnit,
    compressionMethod: CompressionMethod = CompressionMethod.NONE,
    order: ByteOrder = ByteOrder.nativeOrder()
  ): RWRecWriter = {
    if (timePrecision.compareTo(ChronoUnit.MILLIS) >= 0) {
      // Use millisecond format unless greater precision is needed
      forOutputStreamFormat(s, FileFormat.FT_RWIPV6ROUTING, 1, compressionMethod, order)
    } else {
      // Use nanosecond format for anything requesting greater precision
      forOutputStreamFormat(s, FileFormat.FT_RWIPV6ROUTING, 4, compressionMethod, order)
    }
  }

  @deprecated(
    "please use RWRecWriter.forOutputStream family of methods instead for control over output format",
    "netsa-io-silk 1.7.0"
  )
  def toOutputStream(
    s: OutputStream,
    compressionMethod: CompressionMethod = CompressionMethod.NONE
  ): RWRecWriter = forOutputStreamFormat(s, FileFormat.FT_RWIPV6ROUTING, 1, compressionMethod)
}

// @LICENSE_FOOTER@
//
// Mothra 1.7
//
// Copyright 2025 Carnegie Mellon University.
//
// NO WARRANTY. THIS CARNEGIE MELLON UNIVERSITY AND SOFTWARE ENGINEERING INSTITUTE MATERIAL IS
// FURNISHED ON AN "AS-IS" BASIS. CARNEGIE MELLON UNIVERSITY MAKES NO WARRANTIES OF ANY KIND,
// EITHER EXPRESSED OR IMPLIED, AS TO ANY MATTER INCLUDING, BUT NOT LIMITED TO, WARRANTY OF FITNESS
// FOR PURPOSE OR MERCHANTABILITY, EXCLUSIVITY, OR RESULTS OBTAINED FROM USE OF THE MATERIAL.
// CARNEGIE MELLON UNIVERSITY DOES NOT MAKE ANY WARRANTY OF ANY KIND WITH RESPECT TO FREEDOM FROM
// PATENT, TRADEMARK, OR COPYRIGHT INFRINGEMENT.
//
// Licensed under a GNU GPL 2.0-style license, please see LICENSE.txt or contac
// permission@sei.cmu.edu for full terms.
//
// [DISTRIBUTION STATEMENT A] This material has been approved for public release and unlimited
// distribution.  Please see Copyright notice for non-US Government use and distribution.
//
// This Software includes and/or makes use of Third-Party Software each subject to its own license.
//
// DM24-1649
