// Copyright 2025 by Carnegie Mellon University
// See license information in LICENSE.txt

package org.cert.netsa.io.silk
package io

import java.io.{DataInputStream, InputStream}
import java.util.zip.Inflater

/** A [[BufferReader]] which reads zlib-compressed data from SiLK file streams. */
private[silk] class ZlibInputStreamBuffer(inStream: InputStream) extends BufferReader {
  private val in: DataInputStream = inStream match {
    case in: DataInputStream => in
    case _                   => new DataInputStream(inStream)
  }

  private val inflater: Inflater = new Inflater()

  def close(): Unit = {
    in.close()
  }

  def getNextBuffer(): (Array[Byte], Int) = {
    val compressedSize = in.readInt()
    val uncompressedSize = in.readInt()
    val compressedData = Array.ofDim[Byte](compressedSize)
    val uncompressedData = Array.ofDim[Byte](uncompressedSize)
    in.readFully(compressedData)
    inflater.setInput(compressedData, 0, compressedSize)
    val uncompressedSizeResult = inflater.inflate(uncompressedData, 0, uncompressedSize)
    assert(inflater.needsInput())
    assert(uncompressedSizeResult == uncompressedSize)
    inflater.reset()
    (uncompressedData, uncompressedSize)
  }
}

private[silk] object ZlibInputStreamBuffer {
  def apply(inStream: InputStream): ZlibInputStreamBuffer = new ZlibInputStreamBuffer(inStream)
}

// @LICENSE_FOOTER@
//
// Mothra 1.7
//
// Copyright 2025 Carnegie Mellon University.
//
// NO WARRANTY. THIS CARNEGIE MELLON UNIVERSITY AND SOFTWARE ENGINEERING INSTITUTE MATERIAL IS
// FURNISHED ON AN "AS-IS" BASIS. CARNEGIE MELLON UNIVERSITY MAKES NO WARRANTIES OF ANY KIND,
// EITHER EXPRESSED OR IMPLIED, AS TO ANY MATTER INCLUDING, BUT NOT LIMITED TO, WARRANTY OF FITNESS
// FOR PURPOSE OR MERCHANTABILITY, EXCLUSIVITY, OR RESULTS OBTAINED FROM USE OF THE MATERIAL.
// CARNEGIE MELLON UNIVERSITY DOES NOT MAKE ANY WARRANTY OF ANY KIND WITH RESPECT TO FREEDOM FROM
// PATENT, TRADEMARK, OR COPYRIGHT INFRINGEMENT.
//
// Licensed under a GNU GPL 2.0-style license, please see LICENSE.txt or contac
// permission@sei.cmu.edu for full terms.
//
// [DISTRIBUTION STATEMENT A] This material has been approved for public release and unlimited
// distribution.  Please see Copyright notice for non-US Government use and distribution.
//
// This Software includes and/or makes use of Third-Party Software each subject to its own license.
//
// DM24-1649
