// Copyright 2025 by Carnegie Mellon University
// See license information in LICENSE.txt

package org.cert.netsa.io.silk
package io

import java.io.{DataOutputStream, OutputStream}
import java.util.zip.Deflater

/** A [[BufferWriter]] which compresses data using zlib and writes it to a stream in a form
  * compatible with SiLK file streams.
  *
  * @see [[ZlibOutputStreamBuffer$ the companion object]] for more details.
  */
private[silk] class ZlibOutputStreamBuffer(outStream: OutputStream) extends BufferWriter {
  // The output stream
  private val out: DataOutputStream = outStream match {
    case out: DataOutputStream => out
    case _                     => new DataOutputStream(outStream)
  }

  // The compression engine
  private val deflater: Deflater = new Deflater(Deflater.DEFAULT_COMPRESSION, false)

  /** Return the size of a buffer that is guaranteed to be large enough to hold the result of
    * compressing `sourceLen` bytes of data.
    */
  private def compressBound(sourceLen: Int): Int = {
    // Formula taken from body of the compressBound function in the
    // zlib 1.2.11 source code.
    sourceLen + (sourceLen >>> 12) + (sourceLen >>> 14) + (sourceLen >>> 25) + 13
  }

  /** Compress and write any pending data, finalize the compression engine, and flush the output
    * stream.
    */
  def end(): Unit = {
    assert(deflater.needsInput)
    deflater.end()
    out.flush()
  }

  /** Write the first `length` bytes of data in `buffer` to the output stream.
    *
    * @throws java.io.IOException when an error occurs.
    */
  def putBuffer(uncompressedData: Array[Byte], uncompressedSize: Int): Unit =
    if (uncompressedSize > 0) {

      // create buffer to hold the result.  FIXME: Consider moving this
      // array outside the function
      val maxCompressedSize = compressBound(uncompressedSize)
      val compressedData = Array.ofDim[Byte](maxCompressedSize)

      // compress it
      assert(deflater.needsInput())
      deflater.setInput(uncompressedData, 0, uncompressedSize)
      deflater.finish()

      val compressedSize = deflater.deflate(compressedData, 0, maxCompressedSize)

      // write the two sizes and the compressed block
      out.writeInt(compressedSize)
      out.writeInt(uncompressedSize)
      out.write(compressedData, 0, compressedSize)

      assert(deflater.finished())
      assert(deflater.needsInput())
      deflater.reset()
    }
}

/** The ZlibOutputStreamBuffer object provides support for creating an [[ZlibOutputStreamBuffer]].
  */
private[silk] object ZlibOutputStreamBuffer {
  def apply(outStream: OutputStream): ZlibOutputStreamBuffer = {
    new ZlibOutputStreamBuffer(outStream)
  }
}

// @LICENSE_FOOTER@
//
// Mothra 1.7
//
// Copyright 2025 Carnegie Mellon University.
//
// NO WARRANTY. THIS CARNEGIE MELLON UNIVERSITY AND SOFTWARE ENGINEERING INSTITUTE MATERIAL IS
// FURNISHED ON AN "AS-IS" BASIS. CARNEGIE MELLON UNIVERSITY MAKES NO WARRANTIES OF ANY KIND,
// EITHER EXPRESSED OR IMPLIED, AS TO ANY MATTER INCLUDING, BUT NOT LIMITED TO, WARRANTY OF FITNESS
// FOR PURPOSE OR MERCHANTABILITY, EXCLUSIVITY, OR RESULTS OBTAINED FROM USE OF THE MATERIAL.
// CARNEGIE MELLON UNIVERSITY DOES NOT MAKE ANY WARRANTY OF ANY KIND WITH RESPECT TO FREEDOM FROM
// PATENT, TRADEMARK, OR COPYRIGHT INFRINGEMENT.
//
// Licensed under a GNU GPL 2.0-style license, please see LICENSE.txt or contac
// permission@sei.cmu.edu for full terms.
//
// [DISTRIBUTION STATEMENT A] This material has been approved for public release and unlimited
// distribution.  Please see Copyright notice for non-US Government use and distribution.
//
// This Software includes and/or makes use of Third-Party Software each subject to its own license.
//
// DM24-1649
