// Copyright 2025 by Carnegie Mellon University
// See license information in LICENSE.txt

package org.cert.netsa.io.silk
package io
package flow_formats

import java.nio.ByteBuffer
import java.time.temporal.ChronoUnit

import org.cert.netsa.data.net.{IPv4Address, Port, TCPFlags}

import BufferUtil.*

/*
**  FLOWCAP VERSION 2
**
**    uint32_t      sIP;             //  0- 3  Source IP
**    uint32_t      dIP;             //  4- 7  Destination IP
**
**    uint32_t      bytes;           //  8-11  Byte count
**
**    uint32_t      sTime;           // 12-15  Start time as UNIX epoch secs
**
**    uint16_t      elapsed;         // 16-17  Duration of flow in seconds
**    uint16_t      sPort;           // 18-19  Source port
**
**    uint16_t      dPort;           // 20-21  Destination port
**    uint8_t       input;           // 22     SNMP Input
**    uint8_t       output;          // 23     SNMP Output
**
**    uint8_t       pkts[3]          // 24-26  Count of packets
**    uint8_t       proto            // 27     Protocol
**
**    uint8_t       flags            // 28     OR of TCP Flags on all pkts
**    uint8_t       first_flags;     // 29     TOS (ignored)
**
**
**  30 bytes on disk.
 */

private[silk] object FT_FLOWCAP_v2 extends FlowDecoder {
  val recordLength = 30
  val timePrecision = ChronoUnit.SECONDS
  def decode(buffer: ByteBuffer, offset: Int, header: Header): RWRec = {
    val sIP = buffer.getIPv4Address(offset)
    val dIP = buffer.getIPv4Address(offset + 4)
    val bytes = buffer.getByteCount32(offset + 8)
    val startTime = buffer.getTimeSecs(offset + 12)
    val elapsed = buffer.getElapsedSecs16(offset + 16)
    val sPort = buffer.getPort(offset + 18)
    val dPort = buffer.getPort(offset + 20)
    val input = buffer.getSNMPInterface8(offset + 22)
    val output = buffer.getSNMPInterface8(offset + 23)
    val packets = buffer.getPacketCount24(offset + 24)
    val protocol = buffer.getProtocol(offset + 27)
    val flags = buffer.getTCPFlags(offset + 28)
    // unsupplied
    val flowType = FlowType(0)
    val sensor = Sensor(0)
    val initFlags = TCPFlags(0)
    val restFlags = TCPFlags(0)
    val tcpState = TCPState(0)
    val application = Port(0)
    val memo: Short = 0
    val nhIP = IPv4Address(0)
    RWRec(
      startTime, elapsed, sPort, dPort, protocol, flowType, sensor, flags, initFlags, restFlags,
      tcpState, application, memo, input, output, packets, bytes, sIP, dIP, nhIP
    ).maybeClearTCPStateExpanded
  }
}

// @LICENSE_FOOTER@
//
// Mothra 1.7
//
// Copyright 2025 Carnegie Mellon University.
//
// NO WARRANTY. THIS CARNEGIE MELLON UNIVERSITY AND SOFTWARE ENGINEERING INSTITUTE MATERIAL IS
// FURNISHED ON AN "AS-IS" BASIS. CARNEGIE MELLON UNIVERSITY MAKES NO WARRANTIES OF ANY KIND,
// EITHER EXPRESSED OR IMPLIED, AS TO ANY MATTER INCLUDING, BUT NOT LIMITED TO, WARRANTY OF FITNESS
// FOR PURPOSE OR MERCHANTABILITY, EXCLUSIVITY, OR RESULTS OBTAINED FROM USE OF THE MATERIAL.
// CARNEGIE MELLON UNIVERSITY DOES NOT MAKE ANY WARRANTY OF ANY KIND WITH RESPECT TO FREEDOM FROM
// PATENT, TRADEMARK, OR COPYRIGHT INFRINGEMENT.
//
// Licensed under a GNU GPL 2.0-style license, please see LICENSE.txt or contac
// permission@sei.cmu.edu for full terms.
//
// [DISTRIBUTION STATEMENT A] This material has been approved for public release and unlimited
// distribution.  Please see Copyright notice for non-US Government use and distribution.
//
// This Software includes and/or makes use of Third-Party Software each subject to its own license.
//
// DM24-1649
