// Copyright 2025 by Carnegie Mellon University
// See license information in LICENSE.txt

package org.cert.netsa.io.silk
package io
package flow_formats

import java.nio.ByteBuffer
import java.time.{Duration, Instant}
import java.time.temporal.ChronoUnit

import org.cert.netsa.data.net.{IPv4Address, TCPFlags}

import BufferUtil.*

/*
**  FLOWCAP VERSION 3
**
**  in the following: EXPANDED == ((tcp_state & SK_TCPSTATE_EXPANDED) ? 1 : 0)
**
**    uint32_t      sIP;             //  0- 3  Source IP
**    uint32_t      dIP;             //  4- 7  Destination IP
**
**    uint32_t      bytes;           //  8-11  Byte count
**
**    uint32_t      sTime;           // 12-15  Start time as UNIX epoch secs
**
**    uint16_t      elapsed;         // 16-17  Duration of flow in seconds
**    uint16_t      sPort;           // 18-19  Source port
**
**    uint16_t      dPort;           // 20-21  Destination port
**    uint16_t      service_port;    // 22-23  Port reported by flow collector
**
**    uint8_t       input;           // 24     SNMP Input
**    uint8_t       output;          // 25     SNMP Output
**
**    uint8_t       pkts[3]          // 26-28  Count of packets
**    uint8_t       proto            // 29     Protocol
**
**    uint8_t       flags            // 30     EXPANDED==0: All TCP Flags
**                                   //        EXPANDED==1: Flags !1st pkt
**    uint8_t       first_flags;     // 31     EXPANDED==0: 0
**                                   //        EXPANDED==1: TCP Flags 1st pkt
**    uint8_t       tcp_state;       // 32     TCP state machine info
**    uint8_t       time_frac[3];    // 33-35  sTime msec & elapsed msec
**
**
**  36 bytes on disk.
 */

private[silk] object FT_FLOWCAP_v3 extends FlowDecoder {
  val recordLength = 36
  val timePrecision = ChronoUnit.MILLIS
  def decode(buffer: ByteBuffer, offset: Int, header: Header): RWRec = {
    val sIP = buffer.getIPv4Address(offset)
    val dIP = buffer.getIPv4Address(offset + 4)
    val bytes = buffer.getByteCount32(offset + 8)

    val timeFrac = buffer.getInt24BE(offset + 33)
    val startTimeSecs = buffer.getInt(offset + 12)
    val elapsedSecs = buffer.getShort(offset + 16)

    val startTime = Instant
      .ofEpochMilli(Integer.toUnsignedLong(startTimeSecs) * 1000L + ((timeFrac >> 14) & 0x000003ff))
    val elapsed = Duration.ofMillis(
      java.lang.Short.toUnsignedInt(elapsedSecs) * 1000L + ((timeFrac >> 4) & 0x0000003ff)
    )

    val sPort = buffer.getPort(offset + 18)
    val dPort = buffer.getPort(offset + 20)
    val application = buffer.getPort(offset + 22)
    val input = buffer.getSNMPInterface8(offset + 24)
    val output = buffer.getSNMPInterface8(offset + 25)
    val packets = buffer.getPacketCount24(offset + 26)
    val protocol = buffer.getProtocol(offset + 29)
    val tcpState = buffer.getTCPState(offset + 32)
    val (flags, initFlags, restFlags) =
      if (tcpState.expandedFlags) {
        val restFlags = buffer.getTCPFlags(offset + 30)
        val initFlags = buffer.getTCPFlags(offset + 31)
        val flags = TCPFlags((restFlags.toByte | initFlags.toByte).toByte)
        (flags, initFlags, restFlags)
      } else {
        (buffer.getTCPFlags(offset + 30), TCPFlags(0), TCPFlags(0))
      }
    // unsupplied
    val flowType = FlowType(0)
    val sensor = Sensor(0)
    val memo: Short = 0
    val nhIP = IPv4Address(0)
    RWRec(
      startTime, elapsed, sPort, dPort, protocol, flowType, sensor, flags, initFlags, restFlags,
      tcpState, application, memo, input, output, packets, bytes, sIP, dIP, nhIP
    ).maybeClearTCPStateExpanded
  }
}

// @LICENSE_FOOTER@
//
// Mothra 1.7
//
// Copyright 2025 Carnegie Mellon University.
//
// NO WARRANTY. THIS CARNEGIE MELLON UNIVERSITY AND SOFTWARE ENGINEERING INSTITUTE MATERIAL IS
// FURNISHED ON AN "AS-IS" BASIS. CARNEGIE MELLON UNIVERSITY MAKES NO WARRANTIES OF ANY KIND,
// EITHER EXPRESSED OR IMPLIED, AS TO ANY MATTER INCLUDING, BUT NOT LIMITED TO, WARRANTY OF FITNESS
// FOR PURPOSE OR MERCHANTABILITY, EXCLUSIVITY, OR RESULTS OBTAINED FROM USE OF THE MATERIAL.
// CARNEGIE MELLON UNIVERSITY DOES NOT MAKE ANY WARRANTY OF ANY KIND WITH RESPECT TO FREEDOM FROM
// PATENT, TRADEMARK, OR COPYRIGHT INFRINGEMENT.
//
// Licensed under a GNU GPL 2.0-style license, please see LICENSE.txt or contac
// permission@sei.cmu.edu for full terms.
//
// [DISTRIBUTION STATEMENT A] This material has been approved for public release and unlimited
// distribution.  Please see Copyright notice for non-US Government use and distribution.
//
// This Software includes and/or makes use of Third-Party Software each subject to its own license.
//
// DM24-1649
