// Copyright 2025 by Carnegie Mellon University
// See license information in LICENSE.txt

package org.cert.netsa.io.silk
package io
package flow_formats

import java.nio.ByteBuffer
import java.time.Duration
import java.time.temporal.ChronoUnit

import org.cert.netsa.data.net.{IPv4Address, SNMPInterface}

import BufferUtil.*

/*
**  RWAUGWEB VERSION 5
**
**  in the following: EXPANDED == ((tcp_state & SK_TCPSTATE_EXPANDED) ? 1 : 0)
**
**    uint32_t      rflag_stime;     //  0- 3
**    // uint32_t     rest_flags: 8; //        EXPANDED==0:Empty
**                                   //        EXPANDED==1:TCPflags/!1st pkt
**    // uint32_t     is_tcp    : 1; //        always 1 since all flows TCP
**    // uint32_t     src_is_srv: 1; //        1 if sIP is http server
**    // uint32_t     stime     :22; //        Start time:msec offset from hour
**
**    uint8_t       proto_iflags;    //  4     EXPANDED==0:TCPflags/ALL pkts
**                                   //        EXPANDED==1:TCPflags/1st pkt
**    uint8_t       tcp_state;       //  5     TCP state machine info
**    uint16_t      application;     //  6- 7  Indication of type of traffic
**
**    uint32_t      srvport_elapsed; //  8-11
**    // uint32_t      srv_port : 2; //        Server port: 0=80; 1=443; 2=8080
**    // uint32_t      elapsed  :30; //        Duration of the flow
**
**    uint32_t      pkts;            // 12-15  Count of packets
**    uint32_t      bytes;           // 16-19  Count of bytes
**
**    uint32_t      sIP;             // 20-23  Source IP
**    uint32_t      dIP;             // 24-27  Destination IP
**
**    uint16_t      clnt_port;       // 28-29  Client(non-server) port
**
**
**  30 bytes on disk.
 */

private[silk] object FT_RWAUGWEB_v5 extends FlowDecoder {
  val recordLength = 30
  val timePrecision = ChronoUnit.MILLIS
  def decode(buffer: ByteBuffer, offset: Int, header: Header): RWRec = {
    val (startTime, protocol, tcpState, flags, initFlags, restFlags) = buffer
      .decodeTimesFlagsProto(offset, header)
    val application = buffer.getPort(offset + 6)

    val rflagStime = buffer.getInt(offset)
    val srvPortElapsed = buffer.getInt(offset + 8)
    val clientPort = buffer.getPort(offset + 28)
    val serverPort = buffer.decodeWWWPort((srvPortElapsed >> 30) & 0x00000003)
    val sourceIsServer = ((rflagStime >> 22) & 0x00000001) != 0
    val (sPort, dPort) =
      if (sourceIsServer) {
        (serverPort, clientPort)
      } else {
        (clientPort, serverPort)
      }
    val elapsed = Duration.ofMillis((srvPortElapsed & 0x3fffffff).toLong)

    val packets = buffer.getPacketCount32(offset + 12)
    val bytes = buffer.getByteCount32(offset + 16)
    val sIP = buffer.getIPv4Address(offset + 20)
    val dIP = buffer.getIPv4Address(offset + 24)

    val sensor = header.packedSensorId
    val flowType = header.packedFlowTypeId
    val memo: Short = 0
    val input = SNMPInterface(0)
    val output = SNMPInterface(0)
    val nhIP = IPv4Address(0)
    RWRec(
      startTime, elapsed, sPort, dPort, protocol, flowType, sensor, flags, initFlags, restFlags,
      tcpState, application, memo, input, output, packets, bytes, sIP, dIP, nhIP
    )
  }
}

// @LICENSE_FOOTER@
//
// Mothra 1.7
//
// Copyright 2025 Carnegie Mellon University.
//
// NO WARRANTY. THIS CARNEGIE MELLON UNIVERSITY AND SOFTWARE ENGINEERING INSTITUTE MATERIAL IS
// FURNISHED ON AN "AS-IS" BASIS. CARNEGIE MELLON UNIVERSITY MAKES NO WARRANTIES OF ANY KIND,
// EITHER EXPRESSED OR IMPLIED, AS TO ANY MATTER INCLUDING, BUT NOT LIMITED TO, WARRANTY OF FITNESS
// FOR PURPOSE OR MERCHANTABILITY, EXCLUSIVITY, OR RESULTS OBTAINED FROM USE OF THE MATERIAL.
// CARNEGIE MELLON UNIVERSITY DOES NOT MAKE ANY WARRANTY OF ANY KIND WITH RESPECT TO FREEDOM FROM
// PATENT, TRADEMARK, OR COPYRIGHT INFRINGEMENT.
//
// Licensed under a GNU GPL 2.0-style license, please see LICENSE.txt or contac
// permission@sei.cmu.edu for full terms.
//
// [DISTRIBUTION STATEMENT A] This material has been approved for public release and unlimited
// distribution.  Please see Copyright notice for non-US Government use and distribution.
//
// This Software includes and/or makes use of Third-Party Software each subject to its own license.
//
// DM24-1649
