// Copyright 2025 by Carnegie Mellon University
// See license information in LICENSE.txt

package org.cert.netsa.io.silk
package io
package flow_formats

import java.nio.ByteBuffer
import java.time.temporal.ChronoUnit

import org.cert.netsa.data.net.{IPv4Address, SNMPInterface}

import BufferUtil.*

/*
**  RWAUGWEB VERSION 6
**
**  in the following: EXPANDED == ((tcp_state & SK_TCPSTATE_EXPANDED) ? 1 : 0)
**
**    uint64_t      state_flag_stime;//  0- 7
**    // uint64_t     tcp_state : 8; //        TCP state machine info
**    // uint64_t     init_flags: 8; //        EXPANDED==0:TCPflags/ALL pkts
**                                   //        EXPANDED==1:TCPflags/1st pkt
**    // uint64_t     unused    : 2; //        Reserved
**    // uint64_t     srv_port  : 2; //        Server port: 0=80; 1=443; 2=8080
**    // uint64_t     src_is_srv: 1; //        1 if sIP is http server
**    // uint64_t     is_tcp    : 1; //        1 if FLOW is TCP; 0 otherwise
**    // uint64_t     stime     :42; //        Start time:nsec offset from hour
**
**    uint64_t      rflag_etime;     //  8-15
**    // uint64_t     rest_flags: 8; //        EXPANDED==0:Empty
**                                   //        EXPANDED==1:TCPflags/!1st pkt
**    // uint64_t     etime     :56; //        End time:nsec offset from hour
**
**    uint32_t      pkts;            // 16-19  Count of packets
**    uint32_t      bytes;           // 20-23  Count of bytes
**
**    uint32_t      sIP;             // 24-27  Source IP
**    uint32_t      dIP;             // 28-31  Destination IP
**
**    uint16_t      clnt_port;       // 32-33  Client Port
**    uint16_t      application;     // 34-35  Indication of type of traffic
**
**
**  36 bytes on disk.
 */

private[silk] object FT_RWAUGWEB_v6 extends FlowDecoder {
  val recordLength = 36
  val timePrecision = ChronoUnit.NANOS
  def decode(buffer: ByteBuffer, offset: Int, header: Header): RWRec = {
    val (startTime, endTime, protocol, tcpState, flags, initFlags, restFlags) = buffer
      .decodeNanoTimesFlagsProto(offset, header)
    val stateFlagStime = buffer.getLong(offset)
    val packets = buffer.getPacketCount32(offset + 16)
    val bytes = buffer.getByteCount32(offset + 20)
    val sIP = buffer.getIPv4Address(offset + 24)
    val dIP = buffer.getIPv4Address(offset + 28)
    val clientPort = buffer.getPort(offset + 32)
    val application = buffer.getPort(offset + 34)
    val serverPort = buffer.decodeWWWPort(((stateFlagStime >>> 44) & 0x3).toInt)
    val sourceIsServer = ((stateFlagStime >>> 43) & 0x1) != 0
    val (sPort, dPort) =
      if (sourceIsServer) {
        (serverPort, clientPort)
      } else {
        (clientPort, serverPort)
      }
    val sensor = header.packedSensorId
    val flowType = header.packedFlowTypeId
    val memo: Short = 0
    val input = SNMPInterface(0)
    val output = SNMPInterface(0)
    val nhIP = IPv4Address(0)
    RWRec(
      startTime, endTime, sPort, dPort, protocol, flowType, sensor, flags, initFlags, restFlags,
      tcpState, application, memo, input, output, packets, bytes, sIP, dIP, nhIP
    )
  }
}

// @LICENSE_FOOTER@
//
// Mothra 1.7
//
// Copyright 2025 Carnegie Mellon University.
//
// NO WARRANTY. THIS CARNEGIE MELLON UNIVERSITY AND SOFTWARE ENGINEERING INSTITUTE MATERIAL IS
// FURNISHED ON AN "AS-IS" BASIS. CARNEGIE MELLON UNIVERSITY MAKES NO WARRANTIES OF ANY KIND,
// EITHER EXPRESSED OR IMPLIED, AS TO ANY MATTER INCLUDING, BUT NOT LIMITED TO, WARRANTY OF FITNESS
// FOR PURPOSE OR MERCHANTABILITY, EXCLUSIVITY, OR RESULTS OBTAINED FROM USE OF THE MATERIAL.
// CARNEGIE MELLON UNIVERSITY DOES NOT MAKE ANY WARRANTY OF ANY KIND WITH RESPECT TO FREEDOM FROM
// PATENT, TRADEMARK, OR COPYRIGHT INFRINGEMENT.
//
// Licensed under a GNU GPL 2.0-style license, please see LICENSE.txt or contac
// permission@sei.cmu.edu for full terms.
//
// [DISTRIBUTION STATEMENT A] This material has been approved for public release and unlimited
// distribution.  Please see Copyright notice for non-US Government use and distribution.
//
// This Software includes and/or makes use of Third-Party Software each subject to its own license.
//
// DM24-1649
