// Copyright 2025 by Carnegie Mellon University
// See license information in LICENSE.txt

package org.cert.netsa.io.silk
package io
package flow_formats

import java.nio.ByteBuffer
import java.time.temporal.ChronoUnit

import org.cert.netsa.data.net.{Port, TCPFlags}

import BufferUtil.*

/*
**  RWFILTER VERSION 3
**
**    uint32_t      sIP;             //  0- 3  Source IP
**    uint32_t      dIP;             //  4- 7  Destination IP
**
**    uint16_t      sPort;           //  8- 9  Source port
**    uint16_t      dPort;           // 10-11  Destination port
**
**    uint32_t      nhIP;            // 12-15  Router Next Hop IP
**    uint16_t      input;           // 16-17  Router incoming SNMP interface
**    uint16_t      output;          // 18-19  Router outgoing SNMP interface
**
**    uint32_t      sTime;           // 20-23  Start time of flow-epoch secs
**    uint32_t      elapsed;         // 24-27  Duration of flow
**
**    uint32_t      pkts_ftype       // 28-31
**    // uint32_t     pkts      :20; //        Count of packets
**    // uint32_t     pflag     : 1; //        'pkts' requires multiplier?
**    // uint32_t     pad       : 3; //        padding/reserved
**    // uint32_t     flowtype  : 8; //        flow type (class&type)
**
**    uint32_t      bpp;             // 32-35
**    // uint32_t     bPPkt     :14; //        Whole bytes-per-packet
**    // uint32_t     bPPFrac   : 6; //        Fractional bytes-per-packet
**    // uint32_t     pad       :12; //        padding/reserved
**
**    uint16_t      sID;             // 36-37  Sensor ID
**
**    uint8_t       proto;           // 38     IP protocol
**    uint8_t       flags;           // 39     OR of all TCP flags on all pkts
**
**
**  40 bytes on disk.
 */

private[silk] object FT_RWFILTER_v3 extends FlowDecoder {
  val recordLength = 40
  val timePrecision = ChronoUnit.SECONDS
  def decode(buffer: ByteBuffer, offset: Int, header: Header): RWRec = {
    val sIP = buffer.getIPv4Address(offset)
    val dIP = buffer.getIPv4Address(offset + 4)
    val sPort = buffer.getPort(offset + 8)
    val dPort = buffer.getPort(offset + 10)
    val nhIP = buffer.getIPv4Address(offset + 12)
    val input = buffer.getSNMPInterface16(offset + 16)
    val output = buffer.getSNMPInterface16(offset + 18)
    val startTime = buffer.getTimeSecs(offset + 20)
    val elapsed = buffer.getElapsedSecs32(offset + 24)

    val pktsFtype = buffer.getInt(offset + 28)
    val inPackets = (pktsFtype >> 12) & 0x000fffff
    val pFlag = (pktsFtype & 0x00000800) != 0
    val flowType = FlowType(pktsFtype.toByte)
    val bpp = (buffer.getInt(offset + 32) >> 12) & 0x000fffff

    val sensor = buffer.getSensor16(offset + 36)
    val protocol = buffer.getProtocol(offset + 38)
    val flags = buffer.getTCPFlags(offset + 39)

    val (bytes, packets) = buffer.decodeBytesPackets(bpp, inPackets, pFlag)

    val initFlags = TCPFlags(0)
    val restFlags = TCPFlags(0)
    val tcpState = TCPState(0)
    val application = Port(0)
    val memo: Short = 0

    RWRec(
      startTime, elapsed, sPort, dPort, protocol, flowType, sensor, flags, initFlags, restFlags,
      tcpState, application, memo, input, output, packets, bytes, sIP, dIP, nhIP
    )
  }
}

// @LICENSE_FOOTER@
//
// Mothra 1.7
//
// Copyright 2025 Carnegie Mellon University.
//
// NO WARRANTY. THIS CARNEGIE MELLON UNIVERSITY AND SOFTWARE ENGINEERING INSTITUTE MATERIAL IS
// FURNISHED ON AN "AS-IS" BASIS. CARNEGIE MELLON UNIVERSITY MAKES NO WARRANTIES OF ANY KIND,
// EITHER EXPRESSED OR IMPLIED, AS TO ANY MATTER INCLUDING, BUT NOT LIMITED TO, WARRANTY OF FITNESS
// FOR PURPOSE OR MERCHANTABILITY, EXCLUSIVITY, OR RESULTS OBTAINED FROM USE OF THE MATERIAL.
// CARNEGIE MELLON UNIVERSITY DOES NOT MAKE ANY WARRANTY OF ANY KIND WITH RESPECT TO FREEDOM FROM
// PATENT, TRADEMARK, OR COPYRIGHT INFRINGEMENT.
//
// Licensed under a GNU GPL 2.0-style license, please see LICENSE.txt or contac
// permission@sei.cmu.edu for full terms.
//
// [DISTRIBUTION STATEMENT A] This material has been approved for public release and unlimited
// distribution.  Please see Copyright notice for non-US Government use and distribution.
//
// This Software includes and/or makes use of Third-Party Software each subject to its own license.
//
// DM24-1649
