// Copyright 2025 by Carnegie Mellon University
// See license information in LICENSE.txt

package org.cert.netsa.io.silk
package io
package flow_formats

import java.nio.ByteBuffer
import java.time.{Duration, Instant}
import java.time.temporal.ChronoUnit

import BufferUtil.*

/*
**  RWFILTER VERSION 4
**  RWFILTER VERSION 5
**
**  in the following: EXPANDED == ((tcp_state & SK_TCPSTATE_EXPANDED) ? 1 : 0)
**
**    uint32_t      sIP;             //  0- 3  Source IP
**    uint32_t      dIP;             //  4- 7  Destination IP
**
**    uint16_t      sPort;           //  8- 9  Source port
**    uint16_t      dPort;           // 10-11  Destination port
**
**    uint32_t      nhIP;            // 12-15  Router Next Hop IP
**    uint16_t      input;           // 16-17  Router incoming SNMP interface
**    uint16_t      output;          // 18-19  Router outgoing SNMP interface
**
**    uint32_t      sTime;           // 20-23  Start time of flow-epoch secs
**    uint32_t      elapsed;         // 24-27  Duration of flow
**
**    uint32_t      pkts_stimems;    // 28-31
**    // uint32_t     pkts      :20; //        Count of packets
**    // uint32_t     pflag     : 1; //        'pkts' requires multiplier?
**    // uint32_t     is_tcp    : 1; //        1 if flow is TCP; 0 otherwise
**    // uint32_t     sTime_msec:10; //        Fractional sTime (millisec)
**
**    uint32_t      bb_elapsems;      // 32-35
**    // uint32_t     bPPkt     :14; //        Whole bytes-per-packet
**    // uint32_t     bPPFrac   : 6; //        Fractional bytes-per-packet
**    // uint32_t     padding   : 2; //        padding/reserved
**    // uint32_t     elaps_msec:10; //        Fractional elapsed (millisec)
**
**    uint16_t      sID;             // 36-37  Sensor ID
**
**    uint8_t       flowtype;        // 38     flow type (class&type)
**    uint8_t       prot_flags;      // 39     is_tcp==0: IP protocol
**                                   //        is_tcp==1 &&
**                                   //          EXPANDED==0:TCPflags/all pkts
**                                   //          EXPANDED==1:TCPflags/1st pkt
**
**    uint16_t      application;     // 40-41  Generator of traffic
**
**    uint8_t       tcp_state;       // 42     TCP state machine info
**    uint8_t       rest_flags;      // 43     is_tcp==0: Flow's reported flags
**                                   //        is_tcp==1 &&
**                                   //          EXPANDED==0:Empty
**                                   //          EXPANDED==1:TCPflags/!1st pkt
**
**
**  44 bytes on disk.
 */

private[silk] object FT_RWFILTER_v4 extends FlowDecoder {
  val recordLength = 44
  val timePrecision = ChronoUnit.MILLIS
  def decode(buffer: ByteBuffer, offset: Int, header: Header): RWRec = {
    val sIP = buffer.getIPv4Address(offset)
    val dIP = buffer.getIPv4Address(offset + 4)
    val sPort = buffer.getPort(offset + 8)
    val dPort = buffer.getPort(offset + 10)
    val nhIP = buffer.getIPv4Address(offset + 12)
    val input = buffer.getSNMPInterface16(offset + 16)
    val output = buffer.getSNMPInterface16(offset + 18)
    val startTimeSecs = buffer.getInt(offset + 20)
    val elapsedSecs = buffer.getInt(offset + 24)

    val packetsStartTimeMs = buffer.getInt(offset + 28)
    val bbElapsedMs = buffer.getInt(offset + 32)

    val startTime = Instant.ofEpochMilli(
      Integer.toUnsignedLong(startTimeSecs) * 1000L + (packetsStartTimeMs & 0x000003ff)
    )

    val elapsed = Duration
      .ofMillis(Integer.toUnsignedLong(elapsedSecs) * 1000L + (bbElapsedMs & 0x000003ff))

    val inPackets = (packetsStartTimeMs >> 12) & 0x000fffff
    val pFlag = (packetsStartTimeMs & 0x00000800) != 0
    val isTcp = (packetsStartTimeMs & 0x00000400) != 0
    val bpp = (bbElapsedMs >> 12) & 0x000fffff

    val sensor = buffer.getSensor16(offset + 36)
    val flowType = buffer.getFlowType(offset + 38)
    val protFlags = buffer.get(offset + 39)
    val application = buffer.getPort(offset + 40)
    val tcpState = buffer.getTCPState(offset + 42)
    val inRestFlags = buffer.get(offset + 43)

    val (bytes, packets) = buffer.decodeBytesPackets(bpp, inPackets, pFlag)

    val (protocol, flags, initFlags, restFlags) = buffer
      .decodeProtoFlags(isTcp, protFlags, tcpState, inRestFlags)

    val memo: Short = 0

    RWRec(
      startTime, elapsed, sPort, dPort, protocol, flowType, sensor, flags, initFlags, restFlags,
      tcpState, application, memo, input, output, packets, bytes, sIP, dIP, nhIP
    )
  }
}

// @LICENSE_FOOTER@
//
// Mothra 1.7
//
// Copyright 2025 Carnegie Mellon University.
//
// NO WARRANTY. THIS CARNEGIE MELLON UNIVERSITY AND SOFTWARE ENGINEERING INSTITUTE MATERIAL IS
// FURNISHED ON AN "AS-IS" BASIS. CARNEGIE MELLON UNIVERSITY MAKES NO WARRANTIES OF ANY KIND,
// EITHER EXPRESSED OR IMPLIED, AS TO ANY MATTER INCLUDING, BUT NOT LIMITED TO, WARRANTY OF FITNESS
// FOR PURPOSE OR MERCHANTABILITY, EXCLUSIVITY, OR RESULTS OBTAINED FROM USE OF THE MATERIAL.
// CARNEGIE MELLON UNIVERSITY DOES NOT MAKE ANY WARRANTY OF ANY KIND WITH RESPECT TO FREEDOM FROM
// PATENT, TRADEMARK, OR COPYRIGHT INFRINGEMENT.
//
// Licensed under a GNU GPL 2.0-style license, please see LICENSE.txt or contac
// permission@sei.cmu.edu for full terms.
//
// [DISTRIBUTION STATEMENT A] This material has been approved for public release and unlimited
// distribution.  Please see Copyright notice for non-US Government use and distribution.
//
// This Software includes and/or makes use of Third-Party Software each subject to its own license.
//
// DM24-1649
