// Copyright 2025 by Carnegie Mellon University
// See license information in LICENSE.txt

package org.cert.netsa.io.silk
package io
package flow_formats

import java.nio.ByteBuffer
import java.time.temporal.ChronoUnit

import org.cert.netsa.data.net.{Port, TCPFlags}

import BufferUtil.*

/*
**  RWGENERIC VERSION 0
**  RWGENERIC VERSION 1
**
**    uint32_t      sIP;             //  0- 3  Source IP
**    uint32_t      dIP;             //  4- 7  Destination IP
**
**    uint16_t      sPort;           //  8- 9  Source port
**    uint16_t      dPort;           // 10-11  Destination port
**
**    uint8_t       proto;           // 12     IP protocol
**    uint8_t       flags;           // 13     OR of all TCP flags on all pkts
**    uint8_t       input;           // 14     Router incoming SNMP interface
**    uint8_t       output;          // 15     Router outgoing SNMP interface
**
**    uint32_t      nhIP;            // 16-19  Router Next Hop IP
**    uint32_t      sTime;           // 20-23  Start time of flow-epoch secs
**    uint32_t      pkts;            // 24-27  Count of packets
**    uint32_t      bytes;           // 28-31  Count of bytes
**    uint32_t      elapsed;         // 32-35  Duration of flow
**
**    uint8_t       sID;             // 36     Sensor ID
**    uint8_t       padding[3];      // 37-39  Padding
**
**  40 bytes on disk with padding (VERSION 0)
**  37 bytes on disk without padding (VERSION 1)
 */

private[silk] object FT_RWGENERIC_v1 extends FlowDecoder {
  val recordLength = 37
  val timePrecision = ChronoUnit.SECONDS
  def decode(buffer: ByteBuffer, offset: Int, header: Header): RWRec = {
    val sIP = buffer.getIPv4Address(offset)
    val dIP = buffer.getIPv4Address(offset + 4)
    val sPort = buffer.getPort(offset + 8)
    val dPort = buffer.getPort(offset + 10)
    val protocol = buffer.getProtocol(offset + 12)
    val flags = buffer.getTCPFlags(offset + 13)
    val input = buffer.getSNMPInterface8(offset + 14)
    val output = buffer.getSNMPInterface8(offset + 15)
    val nhIP = buffer.getIPv4Address(offset + 16)
    val startTime = buffer.getTimeSecs(offset + 20)
    val packets = buffer.getPacketCount32(offset + 24)
    val bytes = buffer.getByteCount32(offset + 28)
    val elapsed = buffer.getElapsedSecs32(offset + 32)
    val sensor = buffer.getSensor8(offset + 36)
    // unsupplied
    val flowType = FlowType(0)
    val initFlags = TCPFlags(0)
    val restFlags = TCPFlags(0)
    val tcpState = TCPState(0)
    val application = Port(0)
    val memo: Short = 0
    RWRec(
      startTime, elapsed, sPort, dPort, protocol, flowType, sensor, flags, initFlags, restFlags,
      tcpState, application, memo, input, output, packets, bytes, sIP, dIP, nhIP
    ).maybeClearTCPStateExpanded
  }
}

// @LICENSE_FOOTER@
//
// Mothra 1.7
//
// Copyright 2025 Carnegie Mellon University.
//
// NO WARRANTY. THIS CARNEGIE MELLON UNIVERSITY AND SOFTWARE ENGINEERING INSTITUTE MATERIAL IS
// FURNISHED ON AN "AS-IS" BASIS. CARNEGIE MELLON UNIVERSITY MAKES NO WARRANTIES OF ANY KIND,
// EITHER EXPRESSED OR IMPLIED, AS TO ANY MATTER INCLUDING, BUT NOT LIMITED TO, WARRANTY OF FITNESS
// FOR PURPOSE OR MERCHANTABILITY, EXCLUSIVITY, OR RESULTS OBTAINED FROM USE OF THE MATERIAL.
// CARNEGIE MELLON UNIVERSITY DOES NOT MAKE ANY WARRANTY OF ANY KIND WITH RESPECT TO FREEDOM FROM
// PATENT, TRADEMARK, OR COPYRIGHT INFRINGEMENT.
//
// Licensed under a GNU GPL 2.0-style license, please see LICENSE.txt or contac
// permission@sei.cmu.edu for full terms.
//
// [DISTRIBUTION STATEMENT A] This material has been approved for public release and unlimited
// distribution.  Please see Copyright notice for non-US Government use and distribution.
//
// This Software includes and/or makes use of Third-Party Software each subject to its own license.
//
// DM24-1649
