// Copyright 2025 by Carnegie Mellon University
// See license information in LICENSE.txt

package org.cert.netsa.io.silk
package io
package flow_formats

import java.nio.ByteBuffer
import java.time.{Duration, Instant}
import java.time.temporal.ChronoUnit

import BufferUtil.*

/*
**  RWGENERIC VERSION 3
**  RWGENERIC VERSION 4
**
**    uint32_t      sIP;             //  0- 3  Source IP
**    uint32_t      dIP;             //  4- 7  Destination IP
**
**    uint16_t      sPort;           //  8- 9  Source port
**    uint16_t      dPort;           // 10-11  Destination port
**
**    uint32_t      nhIP;            // 12-15  Router Next Hop IP
**    uint16_t      input;           // 16-17  Router incoming SNMP interface
**    uint16_t      output;          // 18-19  Router outgoing SNMP interface
**
**    uint32_t      sTime;           // 20-23  Start time of flow-epoch secs
**    uint32_t      elapsed;         // 24-27  Duration of flow
**
**    uint32_t      pkts;            // 28-31  Count of packets
**    uint32_t      bytes;           // 32-35  Count of bytes
**
**    uint8_t       proto;           // 36     IP protocol
**    uint8_t       flow_type;       // 37     Class & Type info
**    uint16_t      sID;             // 38-39  Sensor ID
**
**    uint8_t       flags;           // 40     OR of all flags (Netflow flags)
**    uint8_t       init_flags;      // 41     TCP flags in first packet
**                                   //        or blank for "legacy" data
**    uint8_t       rest_flags;      // 42     TCP flags on non-initial packet
**                                   //        or blank for "legacy" data
**    uint8_t       tcp_state;       // 43     TCP state machine information
**
**    uint32_t      bpp;             // 44-47  Bytes-per-Packet
**
**    uint16_t      sTime_msec;      // 48-49  Start time fraction (millisec)
**    uint16_t      elapsed_msec;    // 50-51  Elapsed time fraction (millisec)
**
**    uint16_t      application;     // 52-53  Type of traffic
**    uint8_t[2]                     // 54-55  PADDING
**
**  56 bytes on disk.
 */

private[silk] object FT_RWGENERIC_v3 extends FlowDecoder {
  val recordLength = 56
  val timePrecision = ChronoUnit.MILLIS
  def decode(buffer: ByteBuffer, offset: Int, header: Header): RWRec = {
    val sIP = buffer.getIPv4Address(offset)
    val dIP = buffer.getIPv4Address(offset + 4)
    val sPort = buffer.getPort(offset + 8)
    val dPort = buffer.getPort(offset + 10)
    val nhIP = buffer.getIPv4Address(offset + 12)
    val input = buffer.getSNMPInterface16(offset + 16)
    val output = buffer.getSNMPInterface16(offset + 18)
    val startTimeSecs = Integer.toUnsignedLong(buffer.getInt(offset + 20))
    val elapsedSecs = Integer.toUnsignedLong(buffer.getInt(offset + 24))
    val packets = buffer.getPacketCount32(offset + 28)
    val bytes = buffer.getByteCount32(offset + 32)
    val protocol = buffer.getProtocol(offset + 36)
    val flowType = buffer.getFlowType(offset + 37)
    val sensor = buffer.getSensor16(offset + 38)
    val flags = buffer.getTCPFlags(offset + 40)
    val initFlags = buffer.getTCPFlags(offset + 41)
    val restFlags = buffer.getTCPFlags(offset + 42)
    val tcpState = buffer.getTCPState(offset + 43)
    // val bpp = buffer.getInt(offset + 44) // unused
    val startTimeMsecs = java.lang.Short.toUnsignedInt(buffer.getShort(offset + 48))
    val elapsedMsecs = java.lang.Short.toUnsignedInt(buffer.getShort(offset + 50))
    val startTime = Instant.ofEpochMilli(startTimeSecs * 1000 + startTimeMsecs)
    val elapsed = Duration.ofMillis(elapsedSecs * 1000 + elapsedMsecs)
    val application = buffer.getPort(offset + 52)
    // unsupplied
    val memo: Short = 0
    RWRec(
      startTime, elapsed, sPort, dPort, protocol, flowType, sensor, flags, initFlags, restFlags,
      tcpState, application, memo, input, output, packets, bytes, sIP, dIP, nhIP
    ).maybeClearTCPStateExpanded
  }
}

// @LICENSE_FOOTER@
//
// Mothra 1.7
//
// Copyright 2025 Carnegie Mellon University.
//
// NO WARRANTY. THIS CARNEGIE MELLON UNIVERSITY AND SOFTWARE ENGINEERING INSTITUTE MATERIAL IS
// FURNISHED ON AN "AS-IS" BASIS. CARNEGIE MELLON UNIVERSITY MAKES NO WARRANTIES OF ANY KIND,
// EITHER EXPRESSED OR IMPLIED, AS TO ANY MATTER INCLUDING, BUT NOT LIMITED TO, WARRANTY OF FITNESS
// FOR PURPOSE OR MERCHANTABILITY, EXCLUSIVITY, OR RESULTS OBTAINED FROM USE OF THE MATERIAL.
// CARNEGIE MELLON UNIVERSITY DOES NOT MAKE ANY WARRANTY OF ANY KIND WITH RESPECT TO FREEDOM FROM
// PATENT, TRADEMARK, OR COPYRIGHT INFRINGEMENT.
//
// Licensed under a GNU GPL 2.0-style license, please see LICENSE.txt or contac
// permission@sei.cmu.edu for full terms.
//
// [DISTRIBUTION STATEMENT A] This material has been approved for public release and unlimited
// distribution.  Please see Copyright notice for non-US Government use and distribution.
//
// This Software includes and/or makes use of Third-Party Software each subject to its own license.
//
// DM24-1649
