// Copyright 2025 by Carnegie Mellon University
// See license information in LICENSE.txt

package org.cert.netsa.io.silk
package io
package flow_formats

import java.nio.ByteBuffer
import java.time.temporal.ChronoUnit
import scala.annotation.nowarn

import BufferUtil.*

/*
**  RWIPV6ROUTING VERSION 1
**  RWIPV6ROUTING VERSION 2
**
**    int64_t       sTime;       //  0- 7  Flow start time as milliseconds
**                               //        since UNIX epoch
**
**    uint32_t      elapsed;     //  8-11  Duration of flow in milliseconds
**                               //        (Allows for a 49 day flow)
**
**    uint16_t      sPort;       // 12-13  Source port
**    uint16_t      dPort;       // 14-15  Destination port
**
**    uint8_t       proto;       // 16     IP protocol
**    uint8_t       flow_type;   // 17     Class & Type info
**    uint16_t      sID;         // 18-19  Sensor ID
**
**    uint8_t       flags;       // 20     OR of all flags (Netflow flags)
**    uint8_t       init_flags;  // 21     TCP flags in first packet
**                               //        or blank for "legacy" data
**    uint8_t       rest_flags;  // 22     TCP flags on non-initial packet
**                               //        or blank for "legacy" data
**    uint8_t       tcp_state;   // 23     TCP state machine info (below)
**
**    uint16_t      application; // 24-25  Indication of type of traffic
**    uint16_t      memo;        // 26-27  Application specific field
**
**    uint16_t      input;       // 28-29  Router incoming SNMP interface
**    uint16_t      output;      // 30-31  Router outgoing SNMP interface
**
**    uint32_t      pkts;        // 32-35  Count of packets
**    uint32_t      bytes;       // 36-39  Count of bytes
**
**    uint8_t[16]   sIP;         // 40-55  Source IP
**    uint8_t[16]   dIP;         // 56-71  Destination IP
**    uint8_t[16]   nhIP;        // 72-87  Router Next Hop IP
**
**
**  88 bytes on disk.
 */

private[silk] object FT_RWIPV6ROUTING_v1 extends FlowDecoder with FlowEncoder {
  val recordLength = 88
  val timePrecision = ChronoUnit.MILLIS
  def decode(buffer: ByteBuffer, offset: Int, header: Header): RWRec = {
    val startTime = buffer.getTimeMillis(offset)
    val elapsed = buffer.getElapsedMillis(offset + 8)
    val sPort = buffer.getPort(offset + 12)
    val dPort = buffer.getPort(offset + 14)
    val protocol = buffer.getProtocol(offset + 16)
    val flowType = buffer.getFlowType(offset + 17)
    val sensor = buffer.getSensor16(offset + 18)
    val flags = buffer.getTCPFlags(offset + 20)
    val initFlags = buffer.getTCPFlags(offset + 21)
    val restFlags = buffer.getTCPFlags(offset + 22)
    val tcpState = buffer.getTCPState(offset + 23)
    val application = buffer.getApplication(offset + 24)
    val memo = buffer.getMemo(offset + 26)
    val input = buffer.getSNMPInterface16(offset + 28)
    val output = buffer.getSNMPInterface16(offset + 30)
    val packets = buffer.getPacketCount32(offset + 32)
    val bytes = buffer.getByteCount32(offset + 36)
    val sIP = buffer.getIPAddress(offset + 40, tcpState.isIPv6)
    val dIP = buffer.getIPAddress(offset + 56, tcpState.isIPv6)
    val nhIP = buffer.getIPAddress(offset + 72, tcpState.isIPv6)
    RWRec(
      startTime, elapsed, sPort, dPort, protocol, flowType, sensor, flags, initFlags, restFlags,
      tcpState, application, memo, input, output, packets, bytes, sIP, dIP, nhIP
    )
  }
  def encode(
    buffer: ByteBuffer,
    rec: RWRec,
    offset: Int,
    @nowarn("cat=unused")
    header: Header
  ): Unit = {
    buffer
      .putTimeMillis(offset, rec.startTime)
      .putElapsedMillis(offset + 8, rec.elapsed)
      .putPort(offset + 12, rec.sPort)
      .putPort(offset + 14, rec.dPort)
      .putProtocol(offset + 16, rec.protocol)
      .putFlowType(offset + 17, rec.flowType)
      .putSensor16(offset + 18, rec.sensor)
      .putTCPFlags(offset + 20, rec.flags)
      .putTCPFlags(offset + 21, rec.initFlags)
      .putTCPFlags(offset + 22, rec.restFlags)
      .putTCPState(offset + 23, rec.tcpState)
      .putApplication(offset + 24, rec.application)
      .putMemo(offset + 26, rec.memo)
      .putSNMPInterface16(offset + 28, rec.input)
      .putSNMPInterface16(offset + 30, rec.output)
      .putPacketCount32(offset + 32, rec.packets)
      .putByteCount32(offset + 36, rec.bytes)
      .putIPAddress(offset + 40, rec.sIP)
      .putIPAddress(offset + 56, rec.dIP)
      .putIPAddress(offset + 72, rec.nhIP)
    ()
  }
}

// @LICENSE_FOOTER@
//
// Mothra 1.7
//
// Copyright 2025 Carnegie Mellon University.
//
// NO WARRANTY. THIS CARNEGIE MELLON UNIVERSITY AND SOFTWARE ENGINEERING INSTITUTE MATERIAL IS
// FURNISHED ON AN "AS-IS" BASIS. CARNEGIE MELLON UNIVERSITY MAKES NO WARRANTIES OF ANY KIND,
// EITHER EXPRESSED OR IMPLIED, AS TO ANY MATTER INCLUDING, BUT NOT LIMITED TO, WARRANTY OF FITNESS
// FOR PURPOSE OR MERCHANTABILITY, EXCLUSIVITY, OR RESULTS OBTAINED FROM USE OF THE MATERIAL.
// CARNEGIE MELLON UNIVERSITY DOES NOT MAKE ANY WARRANTY OF ANY KIND WITH RESPECT TO FREEDOM FROM
// PATENT, TRADEMARK, OR COPYRIGHT INFRINGEMENT.
//
// Licensed under a GNU GPL 2.0-style license, please see LICENSE.txt or contac
// permission@sei.cmu.edu for full terms.
//
// [DISTRIBUTION STATEMENT A] This material has been approved for public release and unlimited
// distribution.  Please see Copyright notice for non-US Government use and distribution.
//
// This Software includes and/or makes use of Third-Party Software each subject to its own license.
//
// DM24-1649
