// Copyright 2025 by Carnegie Mellon University
// See license information in LICENSE.txt

package org.cert.netsa.io.silk
package io
package flow_formats

import java.nio.ByteBuffer
import java.time.temporal.ChronoUnit

import BufferUtil.*

/*
**  RWIPV6ROUTING VERSION 3
**
**    int64_t       sTime;       //  0- 7  Flow start time as milliseconds
**                               //        since UNIX epoch
**
**    uint32_t      elapsed;     //  8-11  Duration of flow in milliseconds
**                               //        (Allows for a 49 day flow)
**
**    uint16_t      sPort;       // 12-13  Source port
**    uint16_t      dPort;       // 14-15  Destination port
**
**    uint8_t       proto;       // 16     IP protocol
**    uint8_t       flow_type;   // 17     Class & Type info
**    uint16_t      sID;         // 18-19  Sensor ID
**
**    uint8_t       flags;       // 20     OR of all flags (Netflow flags)
**    uint8_t       init_flags;  // 21     TCP flags in first packet
**                               //        or blank for "legacy" data
**    uint8_t       rest_flags;  // 22     TCP flags on non-initial packet
**                               //        or blank for "legacy" data
**    uint8_t       tcp_state;   // 23     TCP state machine info (below)
**
**    uint16_t      application; // 24-25  Indication of type of traffic
**    uint16_t      memo;        // 26-27  Application specific field
**
**    uint32_t      input;       // 28-31  Router incoming SNMP interface
**
**    uint64_t      pkts;        // 32-39  Count of packets
**
**    uint64_t      bytes;       // 40-47  Count of bytes
**
**    uint8_t[16]   sIP;         // 48-63  (IPv4 in 60-63) Source IP
**    uint8_t[16]   dIP;         // 64-79  (IPv4 in 76-79) Destination IP
**    uint8_t[16]   nhIP;        // 80-95  (IPv4 in 92-95) Router Next Hop IP
**
**    uint32_t      output;      // 96-99  Router outgoing SNMP interface
**
**
**  100 bytes on disk.
 */

private[silk] object FT_RWIPV6ROUTING_v3 extends FlowDecoder {
  val recordLength = 100
  val timePrecision = ChronoUnit.MILLIS
  def decode(buffer: ByteBuffer, offset: Int, header: Header): RWRec = {
    val startTime = buffer.getTimeMillis(offset)
    val elapsed = buffer.getElapsedMillis(offset + 8)
    val sPort = buffer.getPort(offset + 12)
    val dPort = buffer.getPort(offset + 14)
    val protocol = buffer.getProtocol(offset + 16)
    val flowType = buffer.getFlowType(offset + 17)
    val sensor = buffer.getSensor16(offset + 18)
    val flags = buffer.getTCPFlags(offset + 20)
    val initFlags = buffer.getTCPFlags(offset + 21)
    val restFlags = buffer.getTCPFlags(offset + 22)
    val tcpState = buffer.getTCPState(offset + 23)
    val application = buffer.getApplication(offset + 24)
    val memo = buffer.getMemo(offset + 26)
    val input = buffer.getSNMPInterface32(offset + 28)
    val packets = buffer.getPacketCount64(offset + 32)
    val bytes = buffer.getByteCount64(offset + 40)
    val sIP = buffer.getIPAddress(offset + 48, tcpState.isIPv6)
    val dIP = buffer.getIPAddress(offset + 64, tcpState.isIPv6)
    val nhIP = buffer.getIPAddress(offset + 80, tcpState.isIPv6)
    val output = buffer.getSNMPInterface32(offset + 96)
    RWRec(
      startTime, elapsed, sPort, dPort, protocol, flowType, sensor, flags, initFlags, restFlags,
      tcpState, application, memo, input, output, packets, bytes, sIP, dIP, nhIP
    )
  }
}

// @LICENSE_FOOTER@
//
// Mothra 1.7
//
// Copyright 2025 Carnegie Mellon University.
//
// NO WARRANTY. THIS CARNEGIE MELLON UNIVERSITY AND SOFTWARE ENGINEERING INSTITUTE MATERIAL IS
// FURNISHED ON AN "AS-IS" BASIS. CARNEGIE MELLON UNIVERSITY MAKES NO WARRANTIES OF ANY KIND,
// EITHER EXPRESSED OR IMPLIED, AS TO ANY MATTER INCLUDING, BUT NOT LIMITED TO, WARRANTY OF FITNESS
// FOR PURPOSE OR MERCHANTABILITY, EXCLUSIVITY, OR RESULTS OBTAINED FROM USE OF THE MATERIAL.
// CARNEGIE MELLON UNIVERSITY DOES NOT MAKE ANY WARRANTY OF ANY KIND WITH RESPECT TO FREEDOM FROM
// PATENT, TRADEMARK, OR COPYRIGHT INFRINGEMENT.
//
// Licensed under a GNU GPL 2.0-style license, please see LICENSE.txt or contac
// permission@sei.cmu.edu for full terms.
//
// [DISTRIBUTION STATEMENT A] This material has been approved for public release and unlimited
// distribution.  Please see Copyright notice for non-US Government use and distribution.
//
// This Software includes and/or makes use of Third-Party Software each subject to its own license.
//
// DM24-1649
