// Copyright 2025 by Carnegie Mellon University
// See license information in LICENSE.txt

package org.cert.netsa.io.silk
package io
package flow_formats

import java.nio.ByteBuffer
import java.time.temporal.ChronoUnit
import scala.annotation.nowarn

import BufferUtil.*

/*
**  RWIPV6ROUTING VERSION 4
**
**    int64_t       sTime;       //  0-  7  Flow start time as nanoseconds
**                               //         since UNIX epoch
**
**    int64_t       eTime;       //  8- 15  Flow end time as nanoseconds
**                               //         since UNIX epoch
**
**    uint16_t      sPort;       // 16- 17  Source port
**    uint16_t      dPort;       // 18- 19  Destination port
**
**    uint8_t       proto;       // 20      IP protocol
**    uint8_t       flow_type;   // 21      Class & Type info
**    uint16_t      sID;         // 22- 23  Sensor ID
**
**    uint8_t       flags;       // 24      OR of all flags (Netflow flags)
**    uint8_t       init_flags;  // 25      TCP flags in first packet
**                               //         or blank for "legacy" data
**    uint8_t       rest_flags;  // 26      TCP flags on non-initial packet
**                               //         or blank for "legacy" data
**    uint8_t       tcp_state;   // 27      TCP state machine info (below)
**
**    uint16_t      application; // 28- 29  Indication of type of traffic
**    uint16_t      memo;        // 30- 31  Application specific field
**
**    uint32_t      input;       // 32- 35  Router incoming SNMP interface
**    uint32_t      output;      // 36- 39  Router outgoing SNMP interface
**
**    uint64_t      pkts;        // 40- 47  Count of packets
**
**    uint64_t      bytes;       // 48- 55  Count of bytes
**
**    uint8_t[16]   sIP;         // 56- 71  (IPv4 in 68-71) Source IP
**    uint8_t[16]   dIP;         // 72- 87  (IPv4 in 84-87) Destination IP
**    uint8_t[16]   nhIP;        // 88-103  (IPv4 in 100-103) Router NextHop IP
**
**
**  104 bytes on disk.
 */

private[silk] object FT_RWIPV6ROUTING_v4 extends FlowDecoder with FlowEncoder {
  val recordLength = 104
  val timePrecision = ChronoUnit.NANOS
  def decode(buffer: ByteBuffer, offset: Int, header: Header): RWRec = {
    val startTime = buffer.getTimeNanos(offset)
    val endTime = buffer.getTimeNanos(offset + 8)
    val sPort = buffer.getPort(offset + 16)
    val dPort = buffer.getPort(offset + 18)
    val protocol = buffer.getProtocol(offset + 20)
    val flowType = buffer.getFlowType(offset + 21)
    val sensor = buffer.getSensor16(offset + 22)
    val flags = buffer.getTCPFlags(offset + 24)
    val initFlags = buffer.getTCPFlags(offset + 25)
    val restFlags = buffer.getTCPFlags(offset + 26)
    val tcpState = buffer.getTCPState(offset + 27)
    val application = buffer.getApplication(offset + 28)
    val memo = buffer.getMemo(offset + 30)
    val input = buffer.getSNMPInterface32(offset + 32)
    val output = buffer.getSNMPInterface32(offset + 36)
    val packets = buffer.getPacketCount64(offset + 40)
    val bytes = buffer.getByteCount64(offset + 48)
    val sIP = buffer.getIPAddress(offset + 56, tcpState.isIPv6)
    val dIP = buffer.getIPAddress(offset + 72, tcpState.isIPv6)
    val nhIP = buffer.getIPAddress(offset + 88, tcpState.isIPv6)
    RWRec(
      startTime, endTime, sPort, dPort, protocol, flowType, sensor, flags, initFlags, restFlags,
      tcpState, application, memo, input, output, packets, bytes, sIP, dIP, nhIP
    )
  }
  def encode(
    buffer: ByteBuffer,
    rec: RWRec,
    offset: Int,
    @nowarn("cat=unused")
    header: Header
  ): Unit = {
    buffer
      .putTimeNanos(offset, rec.startTime)
      .putTimeNanos(offset + 8, rec.endTime)
      .putPort(offset + 16, rec.sPort)
      .putPort(offset + 18, rec.dPort)
      .putProtocol(offset + 20, rec.protocol)
      .putFlowType(offset + 21, rec.flowType)
      .putSensor16(offset + 22, rec.sensor)
      .putTCPFlags(offset + 24, rec.flags)
      .putTCPFlags(offset + 25, rec.initFlags)
      .putTCPFlags(offset + 26, rec.restFlags)
      .putTCPState(offset + 27, rec.tcpState)
      .putApplication(offset + 28, rec.application)
      .putMemo(offset + 30, rec.memo)
      .putSNMPInterface32(offset + 32, rec.input)
      .putSNMPInterface32(offset + 36, rec.output)
      .putPacketCount64(offset + 40, rec.packets)
      .putByteCount64(offset + 48, rec.bytes)
      .putIPAddress(offset + 56, rec.sIP)
      .putIPAddress(offset + 72, rec.dIP)
      .putIPAddress(offset + 88, rec.nhIP)
    ()
  }
}

// @LICENSE_FOOTER@
//
// Mothra 1.7
//
// Copyright 2025 Carnegie Mellon University.
//
// NO WARRANTY. THIS CARNEGIE MELLON UNIVERSITY AND SOFTWARE ENGINEERING INSTITUTE MATERIAL IS
// FURNISHED ON AN "AS-IS" BASIS. CARNEGIE MELLON UNIVERSITY MAKES NO WARRANTIES OF ANY KIND,
// EITHER EXPRESSED OR IMPLIED, AS TO ANY MATTER INCLUDING, BUT NOT LIMITED TO, WARRANTY OF FITNESS
// FOR PURPOSE OR MERCHANTABILITY, EXCLUSIVITY, OR RESULTS OBTAINED FROM USE OF THE MATERIAL.
// CARNEGIE MELLON UNIVERSITY DOES NOT MAKE ANY WARRANTY OF ANY KIND WITH RESPECT TO FREEDOM FROM
// PATENT, TRADEMARK, OR COPYRIGHT INFRINGEMENT.
//
// Licensed under a GNU GPL 2.0-style license, please see LICENSE.txt or contac
// permission@sei.cmu.edu for full terms.
//
// [DISTRIBUTION STATEMENT A] This material has been approved for public release and unlimited
// distribution.  Please see Copyright notice for non-US Government use and distribution.
//
// This Software includes and/or makes use of Third-Party Software each subject to its own license.
//
// DM24-1649
