// Copyright 2025 by Carnegie Mellon University
// See license information in LICENSE.txt

package org.cert.netsa.io.silk
package io
package flow_formats

import java.nio.ByteBuffer
import java.time.temporal.ChronoUnit

import org.cert.netsa.data.net.{IPv4Address, IPv6Address, SNMPInterface}

import BufferUtil.*

/*
**  RWIPV6 VERSION 3
**
**  in the following: EXPANDED == ((tcp_state & SK_TCPSTATE_EXPANDED) ? 1 : 0)
**
**    uint64_t      state_flag_stime;//  0- 7
**    // uint64_t     tcp_state : 8; //        TCP state machine info
**    // uint64_t     pro_iflags: 8; //        is_tcp==0: Protocol; else:
**                                   //          EXPANDED==0:TCPflags/ALL pkts
**                                   //          EXPANDED==1:TCPflags/1st pkt
**    // uint64_t     unused    : 5; //        Reserved
**    // uint64_t     is_tcp    : 1; //        1 if FLOW is TCP; 0 otherwise
**    // uint64_t     stime     :42; //        Start time:nsec offset from hour
**
**    uint64_t      rflag_etime;     //  8-15
**    // uint64_t     rest_flags: 8; //        is_tcp==0: Empty; else
**                                   //          EXPANDED==0:Empty
**                                   //          EXPANDED==1:TCPflags/!1st pkt
**    // uint64_t     etime     :56; //        End time:nsec offset from hour
**
**    uint32_t      pkts;            // 16-19  Count of packets
**    uint32_t      bytes;           // 20-23  Count of bytes
**
**    uint8_t[16]   sIP;             // 24-39  Source IP
**    uint8_t[16]   dIP;             // 40-55  Destination IP
**
**    uint16_t      sport;           // 56-57  Source Port
**    uint16_t      dport;           // 58-59  Destination Port
**
**    uint16_t      application;     // 60-61  Indication of type of traffic
**
**
**  62 bytes on disk.
 */

private[silk] object FT_RWIPV6_v3 extends FlowDecoder {
  val recordLength = 62
  val timePrecision = ChronoUnit.NANOS
  def decode(buffer: ByteBuffer, offset: Int, header: Header): RWRec = {
    val (startTime, endTime, protocol, tcpState, flags, initFlags, restFlags) = buffer
      .decodeNanoTimesFlagsProto(offset, header)
    val packets = buffer.getPacketCount32(offset + 16)
    val bytes = buffer.getByteCount32(offset + 20)
    val sIP = buffer.getIPAddress(offset + 24, tcpState.isIPv6)
    val dIP = buffer.getIPAddress(offset + 40, tcpState.isIPv6)
    val nhIP =
      if (tcpState.isIPv6) IPv6Address.zero
      else IPv4Address.zero
    val sPort = buffer.getPort(offset + 56)
    val dPort = buffer.getPort(offset + 58)
    val application = buffer.getApplication(offset + 60)
    val flowType = header.packedFlowTypeId
    val sensor = header.packedSensorId
    val input = SNMPInterface(0)
    val output = SNMPInterface(0)
    val memo: Short = 0
    RWRec(
      startTime, endTime, sPort, dPort, protocol, flowType, sensor, flags, initFlags, restFlags,
      tcpState, application, memo, input, output, packets, bytes, sIP, dIP, nhIP
    )
  }
}

// @LICENSE_FOOTER@
//
// Mothra 1.7
//
// Copyright 2025 Carnegie Mellon University.
//
// NO WARRANTY. THIS CARNEGIE MELLON UNIVERSITY AND SOFTWARE ENGINEERING INSTITUTE MATERIAL IS
// FURNISHED ON AN "AS-IS" BASIS. CARNEGIE MELLON UNIVERSITY MAKES NO WARRANTIES OF ANY KIND,
// EITHER EXPRESSED OR IMPLIED, AS TO ANY MATTER INCLUDING, BUT NOT LIMITED TO, WARRANTY OF FITNESS
// FOR PURPOSE OR MERCHANTABILITY, EXCLUSIVITY, OR RESULTS OBTAINED FROM USE OF THE MATERIAL.
// CARNEGIE MELLON UNIVERSITY DOES NOT MAKE ANY WARRANTY OF ANY KIND WITH RESPECT TO FREEDOM FROM
// PATENT, TRADEMARK, OR COPYRIGHT INFRINGEMENT.
//
// Licensed under a GNU GPL 2.0-style license, please see LICENSE.txt or contac
// permission@sei.cmu.edu for full terms.
//
// [DISTRIBUTION STATEMENT A] This material has been approved for public release and unlimited
// distribution.  Please see Copyright notice for non-US Government use and distribution.
//
// This Software includes and/or makes use of Third-Party Software each subject to its own license.
//
// DM24-1649
