// Copyright 2025 by Carnegie Mellon University
// See license information in LICENSE.txt

package org.cert.netsa.io.silk
package io
package flow_formats

import java.nio.ByteBuffer
import java.time.temporal.ChronoUnit

import org.cert.netsa.data.net.{IPv4Address, Port, SNMPInterface, TCPFlags}

import BufferUtil.*

/*
**  RWNOTROUTED VERSION 3
**  RWNOTROUTED VERSION 4
**
**    uint32_t      sIP;             //  0- 3  Source IP
**    uint32_t      dIP;             //  4- 7  Destination IP
**
**    uint16_t      sPort;           //  8- 9  Source port
**    uint16_t      dPort;           // 10-11  Destination port
**
**    uint32_t      pkts_stime;      // 12-15
**    // uint32_t     pkts      :20; //        Count of packets
**    // uint32_t     sTime     :12; //        Start time--offset from hour
**
**    uint32_t      bbe;             // 16-19
**    // uint32_t     bPPkt     :14; //        Whole bytes-per-packet
**    // uint32_t     bPPFrac   : 6; //        Fractional bytes-per-packet
**    // uint32_t     elapsed   :12; //        Duration of flow
**
**    uint32_t      msec_flags       // 20-23
**    // uint32_t     sTime_msec:10; //        Fractional sTime (millisec)
**    // uint32_t     elaps_msec:10; //        Fractional elapsed (millisec)
**    // uint32_t     pflag     : 1; //        'pkts' requires multiplier?
**    // uint32_t     is_tcp    : 1; //        1 if flow is TCP; 0 otherwise
**    // uint32_t     padding   : 2; //        padding/reserved
**    // uint32_t     prot_flags: 8; //        is_tcp==0: IP protocol
**                                   //        is_tcp==1: TCP flags
**
**    uint16_t      input;           // 24-25  Router incoming SNMP interface
**
**
**  26 bytes on disk.
 */

private[silk] object FT_RWNOTROUTED_v3 extends FlowDecoder {
  val recordLength = 26
  val timePrecision = ChronoUnit.MILLIS
  def decode(buffer: ByteBuffer, offset: Int, header: Header): RWRec = {
    val sIP = buffer.getIPv4Address(offset)
    val dIP = buffer.getIPv4Address(offset + 4)
    val sPort = buffer.getPort(offset + 8)
    val dPort = buffer.getPort(offset + 10)
    val (startTime, packets, bytes, elapsed, protocol, flags) = buffer
      .getTimeBytesPacketsFlags(offset + 12, offset + 16, offset + 20, false, header)
    val input = buffer.getSNMPInterface16(offset + 24)
    // unsupplied
    val flowType = header.packedFlowTypeId
    val sensor = header.packedSensorId
    val output = SNMPInterface(0)
    val initFlags = TCPFlags(0)
    val restFlags = TCPFlags(0)
    val tcpState = TCPState(0)
    val application = Port(0)
    val memo: Short = 0
    val nhIP = IPv4Address(0)
    RWRec(
      startTime, elapsed, sPort, dPort, protocol, flowType, sensor, flags, initFlags, restFlags,
      tcpState, application, memo, input, output, packets, bytes, sIP, dIP, nhIP
    ).maybeClearTCPStateExpanded
  }
}

// @LICENSE_FOOTER@
//
// Mothra 1.7
//
// Copyright 2025 Carnegie Mellon University.
//
// NO WARRANTY. THIS CARNEGIE MELLON UNIVERSITY AND SOFTWARE ENGINEERING INSTITUTE MATERIAL IS
// FURNISHED ON AN "AS-IS" BASIS. CARNEGIE MELLON UNIVERSITY MAKES NO WARRANTIES OF ANY KIND,
// EITHER EXPRESSED OR IMPLIED, AS TO ANY MATTER INCLUDING, BUT NOT LIMITED TO, WARRANTY OF FITNESS
// FOR PURPOSE OR MERCHANTABILITY, EXCLUSIVITY, OR RESULTS OBTAINED FROM USE OF THE MATERIAL.
// CARNEGIE MELLON UNIVERSITY DOES NOT MAKE ANY WARRANTY OF ANY KIND WITH RESPECT TO FREEDOM FROM
// PATENT, TRADEMARK, OR COPYRIGHT INFRINGEMENT.
//
// Licensed under a GNU GPL 2.0-style license, please see LICENSE.txt or contac
// permission@sei.cmu.edu for full terms.
//
// [DISTRIBUTION STATEMENT A] This material has been approved for public release and unlimited
// distribution.  Please see Copyright notice for non-US Government use and distribution.
//
// This Software includes and/or makes use of Third-Party Software each subject to its own license.
//
// DM24-1649
