// Copyright 2025 by Carnegie Mellon University
// See license information in LICENSE.txt

package org.cert.netsa.io.silk
package io
package flow_formats

import java.nio.ByteBuffer
import java.time.temporal.ChronoUnit

import org.cert.netsa.data.net.{Port, TCPFlags}

import BufferUtil.*

/*
**  RWROUTED VERSION 1
**  RWROUTED VERSION 2
**
**    uint32_t      sIP;             //  0- 3  Source IP
**    uint32_t      dIP;             //  4- 7  Destination IP
**
**    uint32_t      nhIP;            //  8-11  Router Next Hop IP
**
**    uint16_t      sPort;           // 12-23  Source port
**    uint16_t      dPort;           // 14-15  Destination port
**
**    uint32_t      pef;             // 16-19
**    // uint32_t     pkts      :20; //        Count of packets
**    // uint32_t     elapsed   :11; //        Duration of flow
**    // uint32_t     pflag     : 1; //        'pkts' requires multiplier?
**
**    uint32_t      sbb;             // 20-23
**    // uint32_t     sTime     :12; //        Start time--offset from hour
**    // uint32_t     bPPkt     :14; //        Whole bytes-per-packet
**    // uint32_t     bPPFrac   : 6; //        Fractional bytes-per-packet
**
**    uint8_t       proto;           // 24     IP protocol
**    uint8_t       flags;           // 25     OR of all TCP flags on all pkts
**    uint8_t       input;           // 26     Router incoming SNMP interface
**    uint8_t       output;          // 27     Router outgoing SNMP interface
**
**
**  28 bytes on disk.
 */

private[silk] object FT_RWROUTED_v1 extends FlowDecoder {
  val recordLength = 28
  val timePrecision = ChronoUnit.SECONDS
  def decode(buffer: ByteBuffer, offset: Int, header: Header): RWRec = {
    val sIP = buffer.getIPv4Address(offset)
    val dIP = buffer.getIPv4Address(offset + 4)
    val nhIP = buffer.getIPv4Address(offset + 8)
    val sPort = buffer.getPort(offset + 12)
    val dPort = buffer.getPort(offset + 14)
    val (startTime, elapsed, bytes, packets) = buffer.decodeSbbPef(offset + 20, offset + 16, header)
    val protocol = buffer.getProtocol(offset + 24)
    val flags = buffer.getTCPFlags(offset + 25)
    val input = buffer.getSNMPInterface8(offset + 26)
    val output = buffer.getSNMPInterface8(offset + 27)
    // unsupplied
    val flowType = header.packedFlowTypeId
    val sensor = header.packedSensorId
    val initFlags = TCPFlags(0)
    val restFlags = TCPFlags(0)
    val tcpState = TCPState(0)
    val application = Port(0)
    val memo: Short = 0
    RWRec(
      startTime, elapsed, sPort, dPort, protocol, flowType, sensor, flags, initFlags, restFlags,
      tcpState, application, memo, input, output, packets, bytes, sIP, dIP, nhIP
    ).maybeClearTCPStateExpanded
  }
}

// @LICENSE_FOOTER@
//
// Mothra 1.7
//
// Copyright 2025 Carnegie Mellon University.
//
// NO WARRANTY. THIS CARNEGIE MELLON UNIVERSITY AND SOFTWARE ENGINEERING INSTITUTE MATERIAL IS
// FURNISHED ON AN "AS-IS" BASIS. CARNEGIE MELLON UNIVERSITY MAKES NO WARRANTIES OF ANY KIND,
// EITHER EXPRESSED OR IMPLIED, AS TO ANY MATTER INCLUDING, BUT NOT LIMITED TO, WARRANTY OF FITNESS
// FOR PURPOSE OR MERCHANTABILITY, EXCLUSIVITY, OR RESULTS OBTAINED FROM USE OF THE MATERIAL.
// CARNEGIE MELLON UNIVERSITY DOES NOT MAKE ANY WARRANTY OF ANY KIND WITH RESPECT TO FREEDOM FROM
// PATENT, TRADEMARK, OR COPYRIGHT INFRINGEMENT.
//
// Licensed under a GNU GPL 2.0-style license, please see LICENSE.txt or contac
// permission@sei.cmu.edu for full terms.
//
// [DISTRIBUTION STATEMENT A] This material has been approved for public release and unlimited
// distribution.  Please see Copyright notice for non-US Government use and distribution.
//
// This Software includes and/or makes use of Third-Party Software each subject to its own license.
//
// DM24-1649
