// Copyright 2025 by Carnegie Mellon University
// See license information in LICENSE.txt

package org.cert.netsa.io.silk
package io
package ipset

import scala.collection.immutable.Queue
import scala.collection.mutable.ListBuffer

import org.cert.netsa.data.net.{IPBlock, IPv4Address, IPv4Block}

/*
 *    IPSET_REC_VERSION_CLASSC
 *    ========================
 *
 *    The initial format of an IPset file.  This format is only
 *    capable of holding IPv4 addresses.
 *
 *    IPs are stored on disk in blocks of nine 32-bit words which
 *    represent a /24.  The first uint32_t (position [0]) is the base
 *    IP of the /24 (a.b.c.0); the remaining eight uint32_t's are a
 *    256-bit bitmap that represents the IPs in that /24.  Each
 *    uint32_t is a /27.  A high bit indicates the address is present.
 *    The least significant bit in position [1] is a.b.c.0, and the
 *    most significant bit in position [8] is a.b.c.255.
 *
 *    The /24s in the file appear in sorted order.  All values are
 *    written to the file in native byte order.
 *
 *    NOTE: The name is "class c", not "classic", even though it is
 *    the classic version of the IPset format.
 */

private[silk] class IPSetV2Reader(reader: BufferReader, header: Header)
    extends IPSetReader(reader, header) {
  private val recordLength = 36
  private var blocks = Queue[IPBlock]()

  /** A class to combine /24s into larger blocks.
    *
    * The constructor takes an initial IPv4Block that must be a /24.
    *
    * Use addBlock() to add a new IPv4Block which returns true if the block is contiguous with the
    * current range and false otherwise.
    *
    * When addBlock() returns false, the argument to addBlock() has NOT been added to the builder,
    * and the caller should invoke toList as described below.
    *
    * If addBlock() returns true, the caller should use atMaxSize to check whether the maximum
    * number of blocks have been added.
    *
    * When either addBlock() returns false or atMaxSize returns true, the caller should call toList
    * to get a list of IPv4Blocks representing the combined CIDR blocks.
    */
  private class CidrBuilder(ip: IPv4Block) {
    require(ip.prefixLength == 24)

    val prefix = ip.prefixLength
    val minIp = ip.min.toInt
    var maxIp = ip.max.toInt

    val trailing = trailingZero(minIp)
    val maxContiguous = 1 << (trailing - 8)
    val maxCidr = 32 - trailing
    var contiguous = 1

    /** Compute the log base 2 of the integer 'vv'. This is the location of the highest bit. Returns
      * 0 when 'vv' is 0.
      */
    private def logTwo(vv: Int): Int = {
      var v = vv
      var r = 0
      if ((v & 0xffff0000) != 0) {
        v = v >>> 16
        r = r | 16
      }
      if ((v & 0xff00) != 0) {
        v = v >>> 8
        r = r | 8
      }
      if ((v & 0xf0) != 0) {
        v = v >>> 4
        r = r | 4
      }
      if ((v & 0xc) != 0) {
        v = v >>> 2
        r = r | 2
      }
      if ((v & 0x2) != 0) {
        v = v >>> 1
        r = r | 1
      }
      r
    }

    /** Count the number of trailing zeros of the integer 'vv'. Returns 0 when 'vv' is 0. */
    private def trailingZero(vv: Int): Int = {
      // compute the number of trailing 0s.
      if (0 == vv) {
        return 0
      }
      var trailingZero = 1
      var v = vv
      if ((v & 0xffff) == 0) {
        trailingZero = trailingZero + 16
        v = v >>> 16
      }
      if ((v & 0xff) == 0) {
        trailingZero = trailingZero + 8
        v = v >>> 8
      }
      if ((v & 0xf) == 0) {
        trailingZero = trailingZero + 4
        v = v >>> 4
      }
      if ((v & 0x3) == 0) {
        trailingZero = trailingZero + 2
        v = v >>> 2
      }
      trailingZero - (v & 0x1)
    }

    /** Attempt to add newIp to the current CIDR block. The function expects 'newIp' to be an
      * adjacent /24; if it is not, 'newIp' is not added to the CIDR block and the function returns
      * false.
      */
    def addBlock(newIp: IPv4Block): Boolean = {
      if (newIp.prefixLength != 24 || newIp.min.toInt - maxIp != 1) {
        return false
      }
      contiguous = contiguous + 1
      maxIp = newIp.max.toInt
      true
    }

    /** Return true if the block has grown to its largest size and false otherwise. */
    def atMaxSize: Boolean = {
      contiguous == maxContiguous
    }

    /** Return a List of one or more IPv4Blocks representing the combination of all IPBlocks added
      * to the builder.
      */
    def toList: List[IPv4Block] = {
      val buf: ListBuffer[IPv4Block] = new ListBuffer()
      if (maxContiguous == contiguous) {
        buf += IPv4Block(IPv4Address(minIp), maxCidr)
      } else if (1 == contiguous) {
        buf += IPv4Block(IPv4Address(minIp), prefix)
      } else {
        var start = minIp
        do {
          // get position of most significant bit of the number of
          // /24s in the range, and subtract that from 24 to get the
          // CIDR block
          val msb = logTwo(contiguous);
          buf += IPv4Block(IPv4Address(start), 24 - msb)
          start = start + (1 << (8 + msb))
          contiguous = contiguous - (1 << msb)
        } while (contiguous > 0)
      }
      buf.toList
    }
  }

  /** If there are more IPBlocks in 'blocks', do nothing. If 'blocks' is empty and stream is at end
    * of file, do nothing.
    *
    * Otherwise, get the bytes from the buffer that represent a /24 and convert those bytes to one
    * more more IPBlocks and add to 'blocks'. When this function reads a completely full /24, it
    * reads another block from the stream in an attempt to join the /24s into larger blocks.
    */
  private def getMore() = {
    if (blocks.isEmpty && checkAvailable(recordLength)) {
      var builder = None: Option[CidrBuilder]
      var continue = true

      do {
        // read the IP and the contents of the bitmap
        val baseIp = buffer.getInt(bufOffset)
        val bitmapIps = for (pair <- handleBitmap256(buffer, bufOffset + 4)) yield {
          IPv4Block(IPv4Address(baseIp | pair(0)), lenToCidrV4(pair(1)))
        }
        bufOffset = bufOffset + recordLength

        // based on the last block, determine whether to read more
        // data in an attempt to combine multiple /24s into larger
        // blocks.
        val fin = bitmapIps.last
        if (builder == None) {
          // this is the first ipBlock
          if (fin.prefixLength > 24 || 0x100 == (baseIp & 0x100) || !checkAvailable(recordLength)) {
            // cannot be combined with other blocks
            blocks = blocks ++ bitmapIps
            continue = false
          } else {
            builder = Option(new CidrBuilder(fin))
          }
        } else if (builder.get.addBlock(fin)) {
          // this block is contiguous.  check whether the CIDR block
          // is complete or if we are out of data
          if (builder.get.atMaxSize) {
            blocks = blocks ++ builder.get.toList
            continue = false
          } else if (!checkAvailable(recordLength)) {
            blocks = blocks ++ builder.get.toList
            continue = false
          }
        } else {
          // this block is not contiguous with the previous; empty
          // everything in the builder and add to 'blocks'
          blocks = blocks ++ builder.get.toList

          // check whether to start a new builder with the data still
          // in bitmapIps
          if (fin.prefixLength > 24 || 0x100 == (baseIp & 0x100) || !checkAvailable(recordLength)) {
            // cannot start a new builder
            blocks = blocks ++ bitmapIps
            continue = false
          } else {
            builder = Option(new CidrBuilder(fin))
          }
        }
      } while (continue)
    }
  }

  override def containsIPv6: Boolean = false

  override def hasNext: Boolean = {
    getMore()
    !blocks.isEmpty
  }

  override def next(): IPBlock = {
    getMore()
    val ip = blocks.head
    blocks = blocks.tail
    ip
  }
}

// @LICENSE_FOOTER@
//
// Mothra 1.7
//
// Copyright 2025 Carnegie Mellon University.
//
// NO WARRANTY. THIS CARNEGIE MELLON UNIVERSITY AND SOFTWARE ENGINEERING INSTITUTE MATERIAL IS
// FURNISHED ON AN "AS-IS" BASIS. CARNEGIE MELLON UNIVERSITY MAKES NO WARRANTIES OF ANY KIND,
// EITHER EXPRESSED OR IMPLIED, AS TO ANY MATTER INCLUDING, BUT NOT LIMITED TO, WARRANTY OF FITNESS
// FOR PURPOSE OR MERCHANTABILITY, EXCLUSIVITY, OR RESULTS OBTAINED FROM USE OF THE MATERIAL.
// CARNEGIE MELLON UNIVERSITY DOES NOT MAKE ANY WARRANTY OF ANY KIND WITH RESPECT TO FREEDOM FROM
// PATENT, TRADEMARK, OR COPYRIGHT INFRINGEMENT.
//
// Licensed under a GNU GPL 2.0-style license, please see LICENSE.txt or contac
// permission@sei.cmu.edu for full terms.
//
// [DISTRIBUTION STATEMENT A] This material has been approved for public release and unlimited
// distribution.  Please see Copyright notice for non-US Government use and distribution.
//
// This Software includes and/or makes use of Third-Party Software each subject to its own license.
//
// DM24-1649
