// Copyright 2025 by Carnegie Mellon University
// See license information in LICENSE.txt

package org.cert.netsa.io.silk
package io
package ipset

import scala.collection.immutable.Queue

import org.cert.netsa.data.net.{IPBlock, IPv4Address, IPv4Block}

/*
 *    IPSET_REC_VERSION_CIDRBMAP
 *    ==========================
 *
 *    The IPset file format introduced in SiLK-3.7.  The file may
 *    contain either IPv4 or IPv6 addresses (a header entry specifies
 *    which).
 *
 *    IP addresses are stored as "blocks".  Each block starts with an
 *    IP address and a single octet value.  If the single octet is
 *    less than or equal to 0x80, that is the end of the block.  The
 *    IP address is the base address of a netblock and the single
 *    octet is the CIDR mask for the netblock.
 *
 *    If the octet is strictly greater than 0x80, the address is the
 *    base address for a bitmap.  The value of 0x81 means the bitmap
 *    has 256 bits containing a /24 in IPv4 or a /120 in IPv6.  These
 *    bitmaps are similar to those in IPSET_REC_VERSION_CLASSC.
 *
 *    The IPs in the file appear in sorted order.  IPv4 addresses and
 *    values in the bitmap are written in native byte order.  IPv6
 *    addresses are stored as an array of 16 uint8_t's in network byte
 *    order.
 *
 *    The file has a header entry that may be used to determine
 *    whether the file contains IPv4 or IPv6 addresses.  The header
 *    entry is identical with that used by IPSET_REC_VERSION_RADIX.
 *    All fields are 0 except for the leaf length field.  The leaf
 *    length is 4 for files containing IPv4 addresses and 16 for files
 *    containing IPv6.
 */

private[silk] class IPSetV4IPv4Reader(reader: BufferReader, header: Header)
    extends IPSetReader(reader, header) {
  private val ipLength = 4
  private val recordLength = 1 + ipLength
  private var blocks = Queue[IPBlock]()

  /** If there are more IPBlocks in 'blocks', do nothing. Otherwise, read the bytes from 'buffer'
    * that represent an IP address and a prefix. If the prefix is reasonable for the IP address, add
    * a single CIDR block to 'blocks' and return. If the prefix is the special value
    * cidrBitmapFollows, read the 256-bit bitmap and add one or more IPBlocks to 'blocks'. If at end
    * of file, do nothing.
    */
  private def getMore() = {
    if (blocks.isEmpty && checkAvailable(recordLength)) {
      val ip = buffer.getInt(bufOffset)
      val prefix = (0xffL & buffer.get(bufOffset + ipLength)).toInt
      bufOffset = bufOffset + recordLength

      if (prefix != cidrBitmapFollows) {
        blocks = blocks.enqueue(IPv4Block(IPv4Address(ip), prefix))
      } else {
        val L = for (pair <- handleBitmap256(buffer, bufOffset)) yield {
          IPv4Block(IPv4Address(ip | pair(0)), lenToCidrV4(pair(1)))
        }
        blocks = blocks ++ L
        bufOffset = bufOffset + bitmap256Length
      }
    }
  }

  override def containsIPv6: Boolean = false

  override def hasNext: Boolean = {
    getMore()
    !blocks.isEmpty
  }

  override def next(): IPBlock = {
    getMore()
    val ip = blocks.head
    blocks = blocks.tail
    ip
  }
}

// @LICENSE_FOOTER@
//
// Mothra 1.7
//
// Copyright 2025 Carnegie Mellon University.
//
// NO WARRANTY. THIS CARNEGIE MELLON UNIVERSITY AND SOFTWARE ENGINEERING INSTITUTE MATERIAL IS
// FURNISHED ON AN "AS-IS" BASIS. CARNEGIE MELLON UNIVERSITY MAKES NO WARRANTIES OF ANY KIND,
// EITHER EXPRESSED OR IMPLIED, AS TO ANY MATTER INCLUDING, BUT NOT LIMITED TO, WARRANTY OF FITNESS
// FOR PURPOSE OR MERCHANTABILITY, EXCLUSIVITY, OR RESULTS OBTAINED FROM USE OF THE MATERIAL.
// CARNEGIE MELLON UNIVERSITY DOES NOT MAKE ANY WARRANTY OF ANY KIND WITH RESPECT TO FREEDOM FROM
// PATENT, TRADEMARK, OR COPYRIGHT INFRINGEMENT.
//
// Licensed under a GNU GPL 2.0-style license, please see LICENSE.txt or contac
// permission@sei.cmu.edu for full terms.
//
// [DISTRIBUTION STATEMENT A] This material has been approved for public release and unlimited
// distribution.  Please see Copyright notice for non-US Government use and distribution.
//
// This Software includes and/or makes use of Third-Party Software each subject to its own license.
//
// DM24-1649
