// Copyright 2025 by Carnegie Mellon University
// See license information in LICENSE.txt

package org.cert.netsa.io.silk
package io
package ipset

import scala.collection.immutable.Queue

import org.cert.netsa.data.net.{IPBlock, IPv6Address, IPv6Block}

/*
 *    IPSET_REC_VERSION_SLASH64
 *    =========================
 *
 *    The IPset file format introduced in SiLK-3.14.  The file may
 *    contain only IPv6 addresses.
 *
 *    IP addresses are split into two 64 bit values, and they may be
 *    viewed as being on a two level tree, where the upper 64 bits
 *    appear on one level and the lower 64 bits on the other.  Each 64
 *    bit value is followed by a single byte.  When the IP represents
 *    a /64 or larger (more IPs), the block is represented by a single
 *    64 bit value and the prefix; all other IPs are represenetd by
 *    two 64 bit values.  For IPs in the same /64, the upper 64 bit
 *    value appears once followed by 0x82.  Next are the lower 64 bits
 *    of all the IPs in that /64.  Each lower IP is followed either by
 *    a value between 0 and 0x80 indicating the prefix of the netblock
 *    or the value 0x81 indicating that a 256-bit bitmap follows for
 *    the IPs in that /120, similar to IPSET_REC_VERSION_CIDRBMAP and
 *    IPSET_REC_VERSION_CLASSC.
 *
 *    The IPs appear in sorted order.  All values are written in
 *    native byte order.
 *
 *    Although the file format contains only IPv6 addresses, there is
 *    a header entry that confirms this.  The header entry is
 *    identical to that used by IPSET_REC_VERSION_RADIX.  All fields
 *    are 0 except for the leaf length field.  The expected leaf
 *    length is 16.
 */

private[silk] class IPSetV5Reader(reader: BufferReader, header: Header)
    extends IPSetReader(reader, header) {
  import IPSetV5Reader.*
  private val lowerBytesFollow = 0x82

  private var shorts: Array[Short] = Array(0, 0, 0, 0, 0, 0, 0, 0)
  private var state: State = ANY_ALLOWED
  private var blocks = Queue[IPBlock]()

  /** If there are more IPBlocks in 'blocks', do nothing. Otherwise, read the bytes from 'buffer'
    * that represent a 64 bit value and a flag. From the flag, determine if the 64 bit value is a
    * complete IP address (because the lower half of the IP is all 0s), the upper half of an IP
    * address (in which case call read again to get the lower half), the lower half of an IP address
    * with a CIDR prefix, or the lower half of an IP address followed by a 256-bit bitmap.
    *
    * If the value is a single CIDR block, add it to 'blocks' and return. Otherwise, read the
    * 256-bit bitmap and add one or more IPBlocks to 'blocks' and return. If at end of file, do
    * nothing.
    */
  private def getMore(): Unit = {

    def setUpper(v: Long): Unit = {
      shorts = Array(
        ((v >>> 48) & 0xffff).toShort,
        ((v >>> 32) & 0xffff).toShort,
        ((v >>> 16) & 0xffff).toShort,
        (v & 0xffff).toShort,
        0,
        0,
        0,
        0
      )
    }
    def setLower(v: Long): Unit = {
      shorts(4) = ((v >>> 48) & 0xffff).toShort
      shorts(5) = ((v >>> 32) & 0xffff).toShort
      shorts(6) = ((v >>> 16) & 0xffff).toShort
      shorts(7) = (v & 0xffff).toShort
    }

    while (blocks.isEmpty && checkAvailable(9)) {
      val num64 = buffer.getLong(bufOffset)
      val prefix = (0xffL & buffer.get(bufOffset + 8)).toInt
      bufOffset = bufOffset + 9
      if (prefix <= 64) {
        // A CIDR block where a single 64 bit number contains the
        // upper bytes of IPv6, lower bytes are all 0
        if (LOWER_REQUIRED == state || 0 == prefix) {
          throw new SilkDataFormatException(s"Unexpected value for prefix: $prefix")
        }
        state = UPPER_REQUIRED
        setUpper(num64)
        blocks = blocks.enqueue(IPv6Block(IPv6Address(shorts), prefix))
        return

      } else if (prefix <= 128) {
        // A CIDR block where this 64 bit number contains the lower
        // bytes of IPv6; the upper bytes were read previously
        if (UPPER_REQUIRED == state) {
          throw new SilkDataFormatException(s"Unexpected value for prefix: $prefix")
        }
        state = ANY_ALLOWED
        setLower(num64)
        blocks = blocks.enqueue(IPv6Block(IPv6Address(shorts), prefix))
        return

      } else if (prefix == lowerBytesFollow) {
        // The upper 64 bits of an IPv6 address.  Need to loop around
        // to read the lower 64 bits.
        if (LOWER_REQUIRED == state) {
          throw new SilkDataFormatException(s"Unexpected value for prefix: $prefix")
        }
        state = LOWER_REQUIRED
        setUpper(num64)

      } else if (prefix != cidrBitmapFollows) {
        // Invalid value
        throw new SilkDataFormatException(s"Unexpected value for prefix: $prefix")

      } else {
        // Lower 64 bits of IPv6 and a 256-bit bitmap
        if (UPPER_REQUIRED == state) {
          throw new SilkDataFormatException(s"Unexpected value for prefix: $prefix")
        }
        if (!checkAvailable(32)) {
          throw new SilkDataFormatException("Short read")
        }
        state = ANY_ALLOWED
        setLower(num64)
        val L = for (pair <- handleBitmap256(buffer, bufOffset)) yield {
          shorts(7) = ((shorts(7) & 0xff00) | pair(0)).toShort
          IPv6Block(IPv6Address(shorts), lenToCidrV6(pair(1)))
        }
        blocks = blocks ++ L
        bufOffset = bufOffset + bitmap256Length
        return
      }
    }
  }

  override def containsIPv6: Boolean = true

  override def hasNext: Boolean = {
    getMore()
    !blocks.isEmpty
  }

  override def next(): IPBlock = {
    getMore()
    val ip = blocks.head
    blocks = blocks.tail
    ip
  }
}

private[silk] object IPSetV5Reader {
  private sealed trait State
  private case object ANY_ALLOWED extends State
  private case object LOWER_REQUIRED extends State
  private case object UPPER_REQUIRED extends State
}

// @LICENSE_FOOTER@
//
// Mothra 1.7
//
// Copyright 2025 Carnegie Mellon University.
//
// NO WARRANTY. THIS CARNEGIE MELLON UNIVERSITY AND SOFTWARE ENGINEERING INSTITUTE MATERIAL IS
// FURNISHED ON AN "AS-IS" BASIS. CARNEGIE MELLON UNIVERSITY MAKES NO WARRANTIES OF ANY KIND,
// EITHER EXPRESSED OR IMPLIED, AS TO ANY MATTER INCLUDING, BUT NOT LIMITED TO, WARRANTY OF FITNESS
// FOR PURPOSE OR MERCHANTABILITY, EXCLUSIVITY, OR RESULTS OBTAINED FROM USE OF THE MATERIAL.
// CARNEGIE MELLON UNIVERSITY DOES NOT MAKE ANY WARRANTY OF ANY KIND WITH RESPECT TO FREEDOM FROM
// PATENT, TRADEMARK, OR COPYRIGHT INFRINGEMENT.
//
// Licensed under a GNU GPL 2.0-style license, please see LICENSE.txt or contac
// permission@sei.cmu.edu for full terms.
//
// [DISTRIBUTION STATEMENT A] This material has been approved for public release and unlimited
// distribution.  Please see Copyright notice for non-US Government use and distribution.
//
// This Software includes and/or makes use of Third-Party Software each subject to its own license.
//
// DM24-1649
