// Copyright 2025 by Carnegie Mellon University
// See license information in LICENSE.txt

package org.cert.netsa.io.silk
package io
package prefixmap

import scala.collection.immutable.Vector

import org.cert.netsa.data.net.{IPBlock, IPv6Address, IPv6Block}

private[silk] class PrefixMapReaderIPv6(reader: BufferReader, header: Header)
    extends PrefixMapReader[IPBlock](reader, header) {
  // The prefix map as an array of Integers
  private val tree: Array[Int] = readTree()

  // The dictionary of words (empty array for country-code pmap)
  private val dictionary: Array[String] = readDictionary()

  // is this the first time next() is being called?
  private var firstCall = true

  // a variable to trace our path into the tree; initialize it with
  // the left hand side of the root node
  private var path: Vector[Int] = Vector(0)

  override def hasNext: Boolean = {
    path.lastIndexWhere(isEven) != -1
  }

  override def next(): (IPBlock, String) = {
    // the IP address to return
    val ip: Array[Short] = Array.empty.padTo(8, 0.toShort)

    if (firstCall) {
      // this is the first call to next(); no need to do anything
      firstCall = false
    } else {
      // find bottom-most node whose right branch has not been visited
      val n = path.lastIndexWhere(isEven)
      if (-1 == n) {
        if (!path.isEmpty) {
          path = Vector.empty
        }
        throw new NoSuchElementException("PrefixMapReader")
      }

      // remove the bottom-most right branches from path, then replace
      // the bottom-most left value with its right branch
      path = path.take(n) :+ (path(n) + 1)

      // create the current IP address by traversing the path
      val it = path.iterator
      var pos = 0
      var shift = 15
      while (it.hasNext) {
        ip(pos) = (ip(pos) | ((it.next() & 0x1) << shift)).toShort
        if (shift > 0) {
          shift = shift - 1
        } else {
          pos = pos + 1
          shift = 15
        }
      }
    }

    // travel down left hand side until we reach a leaf
    var idx = tree(path.last)
    while (!isLeaf(idx)) {
      path = path :+ idx
      idx = tree(idx)
    }

    // create the IP block to return
    val block = IPv6Block(IPv6Address(ip), path.length)

    if (hasDictionary) {
      (block, dictionary(idx & leafValueMask))
    } else {
      (block, getCountryCode(idx))
    }
  }
}

// @LICENSE_FOOTER@
//
// Mothra 1.7
//
// Copyright 2025 Carnegie Mellon University.
//
// NO WARRANTY. THIS CARNEGIE MELLON UNIVERSITY AND SOFTWARE ENGINEERING INSTITUTE MATERIAL IS
// FURNISHED ON AN "AS-IS" BASIS. CARNEGIE MELLON UNIVERSITY MAKES NO WARRANTIES OF ANY KIND,
// EITHER EXPRESSED OR IMPLIED, AS TO ANY MATTER INCLUDING, BUT NOT LIMITED TO, WARRANTY OF FITNESS
// FOR PURPOSE OR MERCHANTABILITY, EXCLUSIVITY, OR RESULTS OBTAINED FROM USE OF THE MATERIAL.
// CARNEGIE MELLON UNIVERSITY DOES NOT MAKE ANY WARRANTY OF ANY KIND WITH RESPECT TO FREEDOM FROM
// PATENT, TRADEMARK, OR COPYRIGHT INFRINGEMENT.
//
// Licensed under a GNU GPL 2.0-style license, please see LICENSE.txt or contac
// permission@sei.cmu.edu for full terms.
//
// [DISTRIBUTION STATEMENT A] This material has been approved for public release and unlimited
// distribution.  Please see Copyright notice for non-US Government use and distribution.
//
// This Software includes and/or makes use of Third-Party Software each subject to its own license.
//
// DM24-1649
