// Copyright 2015-2022 by Carnegie Mellon University
// See license information in LICENSE.txt

package org.cert.netsa.util

import com.typesafe.scalalogging.LazyLogging
import java.util.Properties
import scala.util.{Failure, Success, Try}

/** Information about versions of CERT NetSA jar files on the classpath,
  * for identification and debugging purposes. This is based on the
  * presence of resources under
  * `/org/cert/netsa/versionInfo/<name>.properties` which are created
  * automatically at build time.
  */
object versionInfo extends LazyLogging {
  private def findKey(name: String, key: String): Option[String] = Try {
    val resource = s"/org/cert/netsa/versionInfo/${name}.properties"
    val props = new Properties()
    for {
      stream <- Option(getClass.getResourceAsStream(resource)) orElse {
        throw new Exception(s"Resource ${resource} not found")
      }
      version <- {
        props.load(stream)
        Option(props.getProperty(key)) orElse {
          throw new Exception(s"Property ${key} not found")
        }
      }
    } yield version
  } match {
    case Success(v) => v
    case Failure(err) =>
      logger.debug(s"Unable to get version property: $err")
      None
  }

  /** A synonym for `version(name)`. */
  def apply(name: String): Option[String] = version(name)

  /**
    * CERT NetSA version information from jar properties. Given the name
    * of a NetSA artifact (such as `"netsa-util"`), returns
    * `Some("<version-number>")` if the version number is found, or
    * `None` if not. For example, `Some("1.5.0")`.
    */
  def version(name: String): Option[String] =
    findKey(name, s"${name}.version")

  /** Detailed NetSA version build information, if available in jar
    * properties. For example,
    * `Some("build-2.12.15-2.4.8-3.3.1-netsa_util")`. While you may be
    * able to deduce some information from this string (for example, the
    * versions of Scala, Spark, and Hadoop used for the build above),
    * the format of this value is not guaranteed, and along with the
    * version number, this should generally be treated as an opaque
    * identifier of the build for debugging purposes.
    */
  def detail(name: String): Option[String] =
    findKey(name, s"${name}.detail")
}

// @LICENSE_FOOTER@
//
// Copyright 2015-2022 Carnegie Mellon University. All Rights Reserved.
//
// This material is based upon work funded and supported by the
// Department of Defense and Department of Homeland Security under
// Contract No. FA8702-15-D-0002 with Carnegie Mellon University for the
// operation of the Software Engineering Institute, a federally funded
// research and development center sponsored by the United States
// Department of Defense. The U.S. Government has license rights in this
// software pursuant to DFARS 252.227.7014.
//
// NO WARRANTY. THIS CARNEGIE MELLON UNIVERSITY AND SOFTWARE ENGINEERING
// INSTITUTE MATERIAL IS FURNISHED ON AN "AS-IS" BASIS. CARNEGIE MELLON
// UNIVERSITY MAKES NO WARRANTIES OF ANY KIND, EITHER EXPRESSED OR
// IMPLIED, AS TO ANY MATTER INCLUDING, BUT NOT LIMITED TO, WARRANTY OF
// FITNESS FOR PURPOSE OR MERCHANTABILITY, EXCLUSIVITY, OR RESULTS
// OBTAINED FROM USE OF THE MATERIAL. CARNEGIE MELLON UNIVERSITY DOES NOT
// MAKE ANY WARRANTY OF ANY KIND WITH RESPECT TO FREEDOM FROM PATENT,
// TRADEMARK, OR COPYRIGHT INFRINGEMENT.
//
// Released under a GNU GPL 2.0-style license, please see LICENSE.txt or
// contact permission@sei.cmu.edu for full terms.
//
// [DISTRIBUTION STATEMENT A] This material has been approved for public
// release and unlimited distribution. Please see Copyright notice for
// non-US Government use and distribution.
//
// Carnegie Mellon(R) and CERT(R) are registered in the U.S. Patent and
// Trademark Office by Carnegie Mellon University.
//
// This software includes and/or makes use of third party software each
// subject to its own license as detailed in LICENSE-thirdparty.tx
//
// DM20-1143
