// Copyright 2015-2022 by Carnegie Mellon University
// See license information in LICENSE.txt

package org.cert.netsa.util

import scala.util.{Failure, Success, Try}

/** An [[scala.collection.Iterator Iterator]] that wraps another
  * [[scala.collection.Iterator Iterator]] with protection for
  * exceptions during iteration. If an exception occurs while calling
  * [[hasNext]] or [[next]], the provided exception handler will be
  * called, and the [[ExceptionGuardedIterator]] will act as if it is
  * now empty.
  *
  * Construction using the companion object will also handle exceptions
  * produced when getting the initial wrapped
  * [[scala.collection.Iterator Iterator]].
  */
class ExceptionGuardedIterator[T]
  (it: Iterator[T], exceptionHandler: Throwable => Unit)
    extends Iterator[T] {

  // These cases are used to wrap the next iterator result up for easy use
  private[this] sealed trait NextResult[+U] {
    def hasNext: Boolean
    def next(): U
  }

  private[this] object NextResult {

    def apply[U](it: Iterator[U]): NextResult[U] =
      Try(it.hasNext) match {
        case Failure(ex) => exceptionHandler(ex); Empty
        case Success(false) => Empty
        case Success(true) =>
          // Call the exception handler for a failure
          Try(it.next) match {
            case Success(v) => Value(v)
            case Failure(ex) => exceptionHandler(ex); Empty
          }
      }

    // We haven't read the next value yet, this should never be used
    case object Unknown extends NextResult[Nothing] {
      override def hasNext: Boolean = throw new AssertionError()
      override def next(): Nothing = throw new AssertionError()
    }

    // We've finished reading the iterator, either due to an exception
    // or to the iterator running out.
    case object Empty extends NextResult[Nothing] {
      override def hasNext: Boolean = false
      override def next(): Nothing =
        throw new NoSuchElementException("next on empty iterator")
    }

    // We have the next value in the iterator ready to go.
    case class Value[+U](value: U) extends NextResult[U] {
      override def hasNext: Boolean = true
      override def next(): U = value
    }

  }

  private[this] var nextResult: NextResult[T] = NextResult.Unknown

  override def hasNext: Boolean = getNextResult().hasNext
  override def next(): T = getNextResult(clear = true).next()

  private[this] def getNextResult(clear: Boolean = false): NextResult[T] =
    if ( nextResult == NextResult.Unknown ) {
      if ( clear ) {
        NextResult(it)
      } else {
        nextResult = NextResult(it)
        nextResult
      }
    } else {
      if ( clear ) {
        val result = nextResult
        nextResult = NextResult.Unknown
        result
      } else {
        nextResult
      }
    }

}

object ExceptionGuardedIterator {
  /** Create an [[scala.collection.Iterator Iterator]] which will call
    * the provided exception handler if any exceptions arise during
    * creation or processing, then behave as if it is an empty
    * [[scala.collection.Iterator Iterator]].
    */
  def apply[T](
    it: => Iterator[T],
    exceptionHandler: Throwable => Unit = { _ => () }):
      ExceptionGuardedIterator[T] = {
    val realIterator = Try(it) match {
      case Failure(ex) =>
        exceptionHandler(ex)
        Iterator.empty
      case Success(iterator) => iterator
    }
    new ExceptionGuardedIterator(realIterator, exceptionHandler)
  }
}

// @LICENSE_FOOTER@
//
// Copyright 2015-2022 Carnegie Mellon University. All Rights Reserved.
//
// This material is based upon work funded and supported by the
// Department of Defense and Department of Homeland Security under
// Contract No. FA8702-15-D-0002 with Carnegie Mellon University for the
// operation of the Software Engineering Institute, a federally funded
// research and development center sponsored by the United States
// Department of Defense. The U.S. Government has license rights in this
// software pursuant to DFARS 252.227.7014.
//
// NO WARRANTY. THIS CARNEGIE MELLON UNIVERSITY AND SOFTWARE ENGINEERING
// INSTITUTE MATERIAL IS FURNISHED ON AN "AS-IS" BASIS. CARNEGIE MELLON
// UNIVERSITY MAKES NO WARRANTIES OF ANY KIND, EITHER EXPRESSED OR
// IMPLIED, AS TO ANY MATTER INCLUDING, BUT NOT LIMITED TO, WARRANTY OF
// FITNESS FOR PURPOSE OR MERCHANTABILITY, EXCLUSIVITY, OR RESULTS
// OBTAINED FROM USE OF THE MATERIAL. CARNEGIE MELLON UNIVERSITY DOES NOT
// MAKE ANY WARRANTY OF ANY KIND WITH RESPECT TO FREEDOM FROM PATENT,
// TRADEMARK, OR COPYRIGHT INFRINGEMENT.
//
// Released under a GNU GPL 2.0-style license, please see LICENSE.txt or
// contact permission@sei.cmu.edu for full terms.
//
// [DISTRIBUTION STATEMENT A] This material has been approved for public
// release and unlimited distribution. Please see Copyright notice for
// non-US Government use and distribution.
//
// Carnegie Mellon(R) and CERT(R) are registered in the U.S. Patent and
// Trademark Office by Carnegie Mellon University.
//
// This software includes and/or makes use of third party software each
// subject to its own license as detailed in LICENSE-thirdparty.tx
//
// DM20-1143
