package org.chrisjaehnen.openlib.testutils.exception;

import static junit.framework.TestCase.assertTrue;
import static junit.framework.TestCase.fail;

/**
 * The ExpectedThrownException class provides a testing framework for validating
 * expected exceptions are thrown.

 * This prevents expected exception limitations in JUnit 4.x as described in
 * <a href="http://jakegoulding.com/blog/2012/09/26/be-careful-when-using-junit-expected-exceptions/">Be careful when using JUnit expected exceptions</a>
 */
public abstract class ExpectedThrownException
{
//----------------------------------------------------------------------------------------------------------------------
// Fields
//----------------------------------------------------------------------------------------------------------------------

    private Class<?> clazz;

//----------------------------------------------------------------------------------------------------------------------
// Constructors
//----------------------------------------------------------------------------------------------------------------------

    /**
     * Required constructor that initializes a new instance of the
     * ExpectedThrownException type with a reference to the class type of the
     * expected exception.
     *
     * @param clazz The class type of the expected exception.
     */
    public ExpectedThrownException(Class<? extends Exception> clazz)
    {
        this.clazz = clazz;
    }

//----------------------------------------------------------------------------------------------------------------------
// Abstract Methods
//----------------------------------------------------------------------------------------------------------------------

    /**
     * Defines an abstract method to implement in subclasses that executes
     * code for a test that throws an expected exception.
     */
    protected abstract void execute();

//----------------------------------------------------------------------------------------------------------------------
// Test Methods
//----------------------------------------------------------------------------------------------------------------------

    /**
     * Executes test code defined in the specified ExpectedThrownException
     * and asserts if the exception was thrown as expected.
     *
     * This method catches a generic {@link java.lang.Exception} and verifies if
     * the expected exception class type was actually thrown. Otherwise,
     * {@link junit.framework.TestCase#fail(String)} is called that fails the
     * test when the expected exception was not thrown.
     */
    public void executeTest()
    {
        try
        {
            execute();
            fail(String.format("The expected exception of type '%s' did not occur. Expected exception failed.", clazz.getName()));
        }
        catch (Exception e)
        {
            if (!clazz.isInstance(e))
            {
                fail(String.format("Encountered unexpected exception of type '%s': '%s'", e.getClass().getName(), e.getMessage()));
            }
            else
            {
                assertTrue(String.format("Expected exception of type '%s' occurred", clazz.getName()), true);
            }
        }
    }
}
