/*
 * The CIP4 Software License, Version 1.0
 *
 *
 * Copyright (c) 2001-2023 The International Cooperation for the Integration of Processes in Prepress, Press and Postpress (CIP4). All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without modification, are permitted provided that the following conditions are met:
 *
 * 1. Redistributions of source code must retain the above copyright notice, this list of conditions and the following disclaimer.
 *
 * 2. Redistributions in binary form must reproduce the above copyright notice, this list of conditions and the following disclaimer in the documentation and/or other materials provided with the
 * distribution.
 *
 * 3. The end-user documentation included with the redistribution, if any, must include the following acknowledgment: "This product includes software developed by the The International Cooperation for
 * the Integration of Processes in Prepress, Press and Postpress (www.cip4.org)" Alternately, this acknowledgment mrSubRefay appear in the software itself, if and wherever such third-party
 * acknowledgments normally appear.
 *
 * 4. The names "CIP4" and "The International Cooperation for the Integration of Processes in Prepress, Press and Postpress" must not be used to endorse or promote products derived from this software
 * without prior written permission. For written permission, please contact info@cip4.org.
 *
 * 5. Products derived from this software may not be called "CIP4", nor may "CIP4" appear in their name, without prior writtenrestartProcesses() permission of the CIP4 organization
 *
 * Usage of this software in commercial products is subject to restrictions. For details please consult info@cip4.org.
 *
 * THIS SOFTWARE IS PROVIDED ``AS IS'' AND ANY EXPRESSED OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED. IN NO EVENT SHALL THE INTERNATIONAL COOPERATION FOR THE INTEGRATION OF PROCESSES IN PREPRESS, PRESS AND POSTPRESS OR ITS CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIrSubRefAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
 * CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE. ====================================================================
 *
 * This software consists of voluntary contributions made by many individuals on behalf of the The International Cooperation for the Integration of Processes in Prepress, Press and Postpress and was
 * originally based on software restartProcesses() copyright (c) 1999-2001, Heidelberger Druckmaschinen AG copyright (c) 1999-2001, Agfa-Gevaert N.V.
 *
 * For more information on The International Cooperation for the Integration of Processes in Prepress, Press and Postpress , please see <http://www.cip4.org/>.
 *
 */
/**
 *
 */
package org.cip4.jdflib.elementwalker;

import java.io.File;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Map;
import java.util.Set;

import org.cip4.jdflib.core.KElement;
import org.cip4.jdflib.ifaces.IElementConverter;
import org.cip4.jdflib.ifaces.IURLSetter;
import org.cip4.jdflib.util.FileUtil;
import org.cip4.jdflib.util.StringUtil;
import org.cip4.jdflib.util.ThreadUtil;
import org.cip4.jdflib.util.UrlUtil;
import org.cip4.jdflib.util.UrlUtil.URLProtocol;

/**
 * @author Rainer Prosi, Heidelberger Druckmaschinen walker that extracts all URLs and dumps them to a directory URLS are modified to reflect the new location
 */
public class URLExtractor extends BaseElementWalker implements IElementConverter
{

	/**
	 * the URL walker
	 *
	 * @author Rainer Prosi, Heidelberger Druckmaschinen
	 */
	public class WalkElement extends BaseWalker
	{
		/**
		 *
		 */
		public WalkElement()
		{
			super(getFactory());
		}

		/**
		 * @see org.cip4.jdflib.elementwalker.BaseWalker#walk(org.cip4.jdflib.core.KElement, org.cip4.jdflib.core.KElement)
		 * @param e
		 * @param trackElem
		 * @return
		 */
		@Override
		public KElement walk(final KElement e, final KElement trackElem)
		{
			return e;
		}
	}

	protected final File dir;
	protected final String baseURL;
	protected Set<URLProtocol> protocols;
	protected final String currentURL;
	protected final Map<String, String> saved;
	protected boolean deleteFile;

	/**
	 * @return the deleteFile
	 */
	public boolean isDeleteFile()
	{
		return deleteFile;
	}

	/**
	 * @param deleteFile the deleteFile to set; if true files are move rather than copied note that files are NOT removed from zip or mime packages
	 */
	public void setDeleteFile(final boolean deleteFile)
	{
		this.deleteFile = deleteFile;
	}

	/**
	 * Getter for list of saved files
	 *
	 * @return the saved
	 */
	public Set<String> getSaved()
	{
		return saved.keySet();
	}

	private boolean wantLog;

	/**
	 * @param dumpDir the local directory where any files are dumped
	 * @param currentURL the current local input url for relative urls - in general this will be a file url (cwd)
	 * @param baseURL the base output url of the extracted data, for instance in an http server environment
	 */
	public URLExtractor(final File dumpDir, final String currentURL, final String baseURL)
	{
		super(new BaseWalkerFactory());
		dir = dumpDir;
		this.baseURL = baseURL;
		this.currentURL = currentURL;
		saved = new HashMap<>();
		protocols = null;
		setDeleteFile(false);
		setWantLog(false);
	}

	/**
	 *
	 *
	 * @param bWant if true, we will log each move
	 *
	 */
	public void setWantLog(final boolean bWant)
	{
		wantLog = bWant;
	}

	/**
	 * add a protocol to the list of protocols that are supported
	 *
	 * @param protocol the protocol to add
	 */
	public void addProtocol(final URLProtocol protocol)
	{
		if (protocols == null)
			protocols = new HashSet<>();
		protocols.add(protocol);
	}

	/**
	 * the resource walker note the naming convention Walkxxx so that it is automagically instantiated by the super classes
	 *
	 * @author prosirai
	 *
	 */
	public class WalkURL extends WalkElement
	{

		/**
		 * fills this into the factory
		 */
		public WalkURL()
		{
			super();
		}

		/**
		 * @see org.cip4.jdflib.elementwalker.BaseWalker#walk(org.cip4.jdflib.core.KElement, org.cip4.jdflib.core.KElement)
		 * @param e the element to walk over
		 * @param trackElem - unused should be null
		 * @return the element to continue walking
		 */
		@Override
		public KElement walk(final KElement e, final KElement trackElem)
		{
			final IURLSetter urlSetter = (IURLSetter) e;
			final String url = StringUtil.getNonEmpty(urlSetter.getURL());
			if (url == null)
			{
				return e;
			}
			String newUrl = saved.get(url);
			if (!StringUtil.isEmpty(newUrl))
			{
				urlSetter.setURL(newUrl);
			}
			// we have a circular reference to something we put here ourselves - no need to do anything
			if (baseURL != null && url.startsWith(baseURL) || newUrl != null)
				return e;

			if (protocols != null)
			{
				final URLProtocol protocol = UrlUtil.getProtocol(url);
				if (!protocols.contains(protocol))
				{
					return e;
				}
			}
			final boolean fileOK = checkFile(url);
			if (fileOK)
			{
				File newFile = UrlUtil.moveToDir(urlSetter, dir, currentURL, true, deleteFile);
				for (int i = 1; i < 4; i++)
				{
					if (newFile != null || UrlUtil.isRelativeURL(url))
						break;
					if (!ThreadUtil.sleep(1234))
					{
						return null;
					}
					newFile = UrlUtil.moveToDir(urlSetter, dir, currentURL, true, deleteFile);
					log.warn("attempting download # " + i + " of URL " + url);
				}
				if (newFile != null)
				{
					if (baseURL != null)
					{
						final String s = UrlUtil.isRelativeURL(url) ? url : UrlUtil.escape(newFile.getName(), false, false);
						final String urlWithDirectory = UrlUtil.getURLWithDirectory(baseURL, s);
						urlSetter.setURL(urlWithDirectory);
					}
					saved.put(url, urlSetter.getURL());
					if (wantLog)
					{
						log.info((deleteFile ? "moved" : "copied ") + url + " to " + urlSetter.getURL());
					}
				}
				else
				{
					log.warn((deleteFile ? "could not move " : "could not copy ") + url + " to " + dir);
					saved.put(url, "");
				}
			}
			return e; // continue
		}

		protected boolean checkFile(final String url)
		{
			if (!UrlUtil.isFile(url) && !UrlUtil.isUNC(url))
			{
				return true;
			}
			File f = UrlUtil.urlToFile(url);
			if (currentURL != null && UrlUtil.isRelativeURL(url))
			{
				f = FileUtil.cleanDots(new File(currentURL, f.getPath()));
			}
			if (f != null)
			{
				if (!f.exists())
				{
					final File parentDir = f.getParentFile();
					if (parentDir != null && !parentDir.exists())
					{
						log.warn("No such parent directory: " + parentDir);
					}
					else if (parentDir != null && !parentDir.canRead())
					{
						log.warn("Cannot read parent directory: " + parentDir);
					}
					log.warn("No such file: " + f);
					return false;
				}
				if (!f.canRead())
				{
					log.warn("Cannot read file: " + f);
					return false;
				}
				else if (f.isDirectory())
				{
					log.warn("Cannot copy directory: " + f);
					return false;
				}
				return true;
			}
			else
			{
				return false;
			}
		}

		/**
		 * @see org.cip4.jdflib.elementwalker.BaseWalker#matches(org.cip4.jdflib.core.KElement)
		 * @param toCheck
		 * @return true if matches
		 */
		@Override
		public boolean matches(final KElement toCheck)
		{
			return (toCheck instanceof IURLSetter);
		}

	}

	/**
	 * @see org.cip4.jdflib.ifaces.IElementConverter#convert(org.cip4.jdflib.core.KElement)
	 */
	@Override
	public KElement convert(final KElement root)
	{
		walkTree(root, null);
		return root;
	}

	/**
	 * @see org.cip4.jdflib.elementwalker.ElementWalker#toString()
	 */
	@Override
	public String toString()
	{
		return super.toString() + protocols + " baseURL: " + baseURL + " currentURL: " + currentURL + " dir: " + dir + " delete: " + deleteFile;
	}
}
