package org.cip4.lib.xjdf.xml;

import jakarta.xml.bind.JAXBContext;
import jakarta.xml.bind.JAXBException;
import jakarta.xml.bind.Marshaller;
import jakarta.xml.bind.Unmarshaller;
import org.cip4.lib.xjdf.exception.XJdfInitException;
import org.cip4.lib.xjdf.exception.XJdfParseException;
import org.cip4.lib.xjdf.xml.internal.JAXBContextFactory;
import org.glassfish.jaxb.runtime.marshaller.NamespacePrefixMapper;

import java.io.ByteArrayInputStream;
import java.io.ByteArrayOutputStream;
import java.io.InputStream;
import java.io.OutputStream;
import java.io.OutputStreamWriter;
import java.nio.charset.Charset;
import java.nio.charset.StandardCharsets;

/**
 * This class is a parser based on the XJDF Schema.
 *
 * @param <T> The root node this parser applies to. Valid values are 'XJDF' and 'XJMF'
 */
public class XJdfParser<T> {

    /**
     * Context for JAXB-handling.
     */
    private final JAXBContext jaxbContext;

    /**
     * Character encoding for xml files.
     */
    private static final Charset CHARSET = StandardCharsets.UTF_8;

    /**
     * Default constructor.
     */
    public XJdfParser() throws XJdfInitException {
        this.jaxbContext = JAXBContextFactory.getInstance();
    }

    /**
     * Read an XML byte array and converts it to an object tree.
     *
     * @param bytes The XML byte array to be parsed.
     * @return The parsed object tree.
     */
    @SuppressWarnings("unchecked")
    public final T readXml(final byte[] bytes) throws XJdfParseException {
        if(bytes == null) {
            throw new XJdfParseException("Cannot read an XML byte array of null.");
        }

        return readXml(new ByteArrayInputStream(bytes));
    }

    /**
     * Read an XML input stream and converts it to an object tree.
     *
     * @param inputStream The XML input stream to be parsed.
     * @return The parsed object tree.
     */
    @SuppressWarnings("unchecked")
    public final T readXml(final InputStream inputStream) throws XJdfParseException {
        try {
            Unmarshaller u = jaxbContext.createUnmarshaller();
            return (T) u.unmarshal(inputStream);

        } catch (JAXBException jaxbException) {
            throw new XJdfParseException(jaxbException);
        }
    }

    /**
     * Write an XJDF object tree to a binary array.
     *
     * @param node Object tree to be written.
     * @return The XJDF Document as xml byte array.
     */
    public final byte[] writeXml(final T node)
        throws XJdfParseException {

        try {
            ByteArrayOutputStream bos = new ByteArrayOutputStream();
            writeXml(node, bos);
            bos.close();

            return bos.toByteArray();

        } catch (Exception exception) {
            throw new XJdfParseException(exception);
        }
    }

    /**
     * Write an XJDF object tree to a binary output stream.
     *
     * @param node         Object tree to be written.
     * @param outputStream The output stream the XML stream is written to.
     */
    public final void writeXml(final T node, final OutputStream outputStream)
        throws XJdfParseException {

        try {
            Marshaller m = createMarshaller();
            m.setProperty(Marshaller.JAXB_FORMATTED_OUTPUT, true);
            m.setProperty(Marshaller.JAXB_ENCODING, CHARSET.name());
            m.setProperty("org.glassfish.jaxb.xmlHeaders", getXmlHeader());

            OutputStreamWriter writer = new OutputStreamWriter(outputStream, CHARSET);
            m.marshal(node, writer);
        } catch (Exception exception) {
            throw new XJdfParseException(exception);
        }
    }

    private String getXmlHeader() {
        String header = "<!-- Generated by CIP4 xJdfLib " + XJdfConstants.XJDFLIB_VERSION + " -->\r\n";
        header = header.replaceAll("  ", " ");
        return header;
    }

    /**
     * Creates and returns a new marshaller object.
     *
     * @return New Marshaller object.
     */
    Marshaller createMarshaller() throws XJdfParseException {
        Marshaller m;

        try {
            // create marshaller
            m = jaxbContext.createMarshaller();
            m.setProperty("org.glassfish.jaxb.namespacePrefixMapper", new XJdfNamespacePrefixMapper());

        } catch (JAXBException jaxbException) {
            throw new XJdfParseException(jaxbException);
        }

        // return marshaller
        return m;
    }

    /**
     * XJDF Namespace Prefix Mapper class for organizing namespace prefixes.
     */
    static class XJdfNamespacePrefixMapper extends NamespacePrefixMapper {

        /**
         * Default constructor.
         */
        public XJdfNamespacePrefixMapper() {
        }

        /**
         * @see org.glassfish.jaxb.runtime.marshaller.NamespacePrefixMapper#getPreferredPrefix(java.lang.String, java.lang.String, boolean)
         */
        @Override
        public String getPreferredPrefix(String namespaceUri, String suggestion, boolean requirePrefix) {
            String result;

            if (requirePrefix) {
                if (namespaceUri.equals(XJdfConstants.NAMESPACE_JDF20)) {
                    result = "xjdf";
                } else {
                    // other namespace
                    result = suggestion;
                }
            } else {
                result = "";
            }

            // return result
            return result;
        }
    }
}
