/*
 * Copyright 2005 Tim Telcik, Werner Guttmann, Ralf Joachim
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.castor.cache.distributed;

import java.util.Map;
import java.util.Properties;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.castor.cache.CacheAcquireException;

/**
 * Tangosol Coherence implementation of Castor JDO Cache.
 * 
 * For more details of Coherence, see http://www.tangosol.com/coherence.jsp 
 * 
 * @param <K> the type of keys maintained by this cache
 * @param <V> the type of cached values
 * 
 * @see <a href="http://www.tangosol.com/coherence.jsp">Coherence Overview</a>  
 * @author <a href="mailto:ttelcik AT hbf DOT com DOT au">Tim Telcik</a>
 * @author <a href="mailto:werner DOT guttmann AT gmx DOT net">Werner Guttmann</a>
 * @author <a href="mailto:ralf DOT joachim AT syscon DOT eu">Ralf Joachim</a>
 * @version $Revision: 9041 $ $Date: 2011-08-16 11:51:17 +0200 (Di, 16 Aug 2011) $
 * @since 1.0
 */
public final class CoherenceCache<K, V> extends AbstractDistributedCache<K, V> {
    //--------------------------------------------------------------------------

    /** The <a href="http://jakarta.apache.org/commons/logging/">Jakarta Commons
     *  Logging </a> instance used for all logging. */
    private static final Log LOG = LogFactory.getLog(CoherenceCache.class);
    
    /** The type of the cache. */
    public static final String TYPE = "coherence";
    
    /** The classname of the implementations factory class. */
    public static final String IMPLEMENTATION = "com.tangosol.net.CacheFactory";
    
    /** Parameter types for calling getCache() method on IMPLEMENTATION. */
    private static final Class<?>[] TYPES_GET_CACHE = new Class[] {String.class};
    
    //--------------------------------------------------------------------------
    // operations for life-cycle management of cache
    
    /**
     * {@inheritDoc}
     */
    public void initialize(final Properties params) throws CacheAcquireException {
        initialize(IMPLEMENTATION, params);
    }

    /**
     * Normally called to initialize CoherenceCache. To be able to test the method
     * without having <code>com.tangosol.net.CacheFactory</code> implementation,
     * it can also be called with a test implementations classname.
     * 
     * @param implementation Cache implementation classname to initialize.
     * @param params Parameters to initialize the cache (e.g. name, capacity).
     * @throws CacheAcquireException If cache can not be initialized.
     */
    @SuppressWarnings("unchecked")
    public void initialize(final String implementation, final Properties params)
    throws CacheAcquireException {
        super.initialize(params);
        
        try {
            ClassLoader ldr = this.getClass().getClassLoader();
            Class<?> cls = ldr.loadClass(implementation);
            setCache((Map<K, V>) invokeStaticMethod(
                    cls, "getCache", TYPES_GET_CACHE, new Object[] {getName()}));
        } catch (Exception e) {
            String msg = "Error creating Coherence cache: " + e.getMessage();
            LOG.error(msg, e);
            throw new CacheAcquireException(msg, e);
        }
    }

    /**
     * {@inheritDoc}
     */
    public void close() {
        super.close();
        
        if (getCache() != null) {
            try {
                invokeMethod(getCache(), "release", null, null);
            } catch (Exception e) {
                String msg = "Error closing Coherence cache: " + e.getMessage();
                LOG.error(msg, e);
            }
        }
    }

    //--------------------------------------------------------------------------
    // getters/setters for cache configuration

    /**
     * {@inheritDoc}
     */
    public String getType() { return TYPE; }

    //--------------------------------------------------------------------------
}
