/*
 * Copyright 2010 Werner Guttmann
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.castor.cpa.jpa.processors.classprocessors;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.castor.core.annotationprocessing.AnnotationTargetException;
import org.castor.core.nature.BaseNature;
import org.castor.cpa.jpa.natures.JPAClassNature;
import org.castor.cpa.jpa.processors.BaseJPAAnnotationProcessor;

import javax.persistence.NamedQueries;
import javax.persistence.NamedQuery;
import java.lang.annotation.Annotation;
import java.lang.reflect.AnnotatedElement;
import java.util.HashMap;
import java.util.Map;

/**
 * Processes the JPA annotation {@link NamedQuery}. After this processor is
 * done, {@link JPAClassNature#getNamedQuery()} returns a valid value!
 * 
 * @author <a href="mailto:wguttmn AT codehaus DOT org">Werner Guttmann</a>
 * @version $Revision: 8994 $ $Date: 2011-08-02 01:40:59 +0200 (Di, 02 Aug 2011) $
 */
public class JPANamedQueriesProcessor extends BaseJPAAnnotationProcessor {
    //-----------------------------------------------------------------------------------
    
    /** The <a href="http://jakarta.apache.org/commons/logging/">Jakarta Commons
     *  Logging </a> instance used for all logging. */
    private final Log _log = LogFactory.getLog(getClass());

    //-----------------------------------------------------------------------------------
    
    /**
     * {@inheritDoc}
     * 
     * @see org.castor.core.annotationprocessing.AnnotationProcessor#
     *      forAnnotationClass()
     */
    public Class<? extends Annotation> forAnnotationClass() {
        return NamedQueries.class;
    }

    /**
     * {@inheritDoc}
     * 
     * @see org.castor.core.annotationprocessing.TargetAwareAnnotationProcessor#
     *      processAnnotation(BaseNature, Annotation, AnnotatedElement)
     */
    public <I extends BaseNature, A extends Annotation> boolean processAnnotation(
            final I info, final A annotation, final AnnotatedElement target)
            throws AnnotationTargetException {

        if ((info instanceof JPAClassNature)
                && (annotation instanceof NamedQueries)
                && (target instanceof Class<?>)
                && (target.isAnnotationPresent(NamedQueries.class))) {
            _log.debug("Processing class annotation " + annotation.toString());
            final JPAClassNature jpaClassNature = (JPAClassNature) info;
            final NamedQueries namedQueries = (NamedQueries) annotation;
            final Map<String, String> namedQueryMap = new HashMap<String, String>();
            NamedQuery[] namedQueriesValues = namedQueries.value();
            if (namedQueriesValues != null && namedQueriesValues.length > 0) {
                for (NamedQuery namedQuery : namedQueriesValues) {
                    namedQueryMap.put(namedQuery.name(), namedQuery.query());
                }
            }
            jpaClassNature.setNamedQuery(namedQueryMap);
            return true;
        }
        return false;
    }
    
    //-----------------------------------------------------------------------------------
}
