/*
 * Copyright 2008 Udai Gupta, Ralf Joachim
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.castor.cpa.query.castorql;

import java.io.ByteArrayInputStream;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.io.UnsupportedEncodingException;

import org.castor.cpa.query.AbstractParser;
import org.castor.cpa.query.ParseException;
import org.castor.cpa.query.TokenManagerError;
import org.castor.cpa.query.QueryObject;

/**
 * Class that implements abstract parser generator.
 * 
 * @author <a href="mailto:mailtoud AT gmail DOT com">Udai Gupta</a>
 * @author <a href="mailto:ralf DOT joachim AT syscon DOT eu">Ralf Joachim</a>
 * @version $Revision: 8994 $ $Date: 2011-08-02 01:40:59 +0200 (Di, 02 Aug 2011) $
 * @since 1.3
 */
public final class CastorQLParserAdapter extends AbstractParser {

    // --------------------------------------------------------------------------

    /**
     * {@inheritDoc}
     * 
     * @throws UnsupportedEncodingException
     * @throws ParseException
     */
    public QueryObject parse(final String oql)
            throws UnsupportedEncodingException, ParseException {
        CastorQLParser parser = null;
        CastorQLParserTokenManager tkmgr = null;
        try {

            tkmgr = createTkmgr(oql);
            parser = new CastorQLParser(tkmgr);
            SimpleNode root = parser.castorQL();
            CastorQLTreeWalker tw = new CastorQLTreeWalker(root);
            return tw.getSelect();

        } catch (org.castor.cpa.query.castorql.ParseException e) {
            parser.ReInit(tkmgr);
            throw new ParseException(e);
        } catch (org.castor.cpa.query.castorql.TokenMgrError e) {
            // parser.ReInit(tkmgr);
            throw new TokenManagerError(e);
        }

    }

    // --------------------------------------------------------------------------

    /**
     * Gets the root SimpleNode instance of JJTREE.
     * 
     * @param oql the query string
     * @return the root SimpleNode of JJTREE 
     * @throws UnsupportedEncodingException the unsupported encoding exception of query string
     * @throws ParseException the QL parse exception
     */
    public SimpleNode getSimpleNode(final String oql)
            throws UnsupportedEncodingException, ParseException {
        CastorQLParser parser = null;
        CastorQLParserTokenManager tkmgr = null;
        try {

            tkmgr = createTkmgr(oql);
            parser = new CastorQLParser(tkmgr);
            return parser.castorQL();
        } catch (org.castor.cpa.query.castorql.ParseException e) {
            parser.ReInit(tkmgr);
            throw new ParseException(e);
        } catch (org.castor.cpa.query.castorql.TokenMgrError e) {
            // parser.ReInit(tkmgr);
            throw new TokenManagerError(e);
        }

    }
    
    // --------------------------------------------------------------------------

    /**
     * Creates the TokenManager instance.
     * 
     * @param oql the query string
     * @return the castor ql parser token manager
     * @throws UnsupportedEncodingException the unsupported encoding exception
     */
    public CastorQLParserTokenManager createTkmgr(final String oql)
            throws UnsupportedEncodingException {

        StringBuffer stringBuffer = new StringBuffer(oql);
        InputStream bis = new ByteArrayInputStream(stringBuffer.toString()
                .getBytes("UTF-8"));
        InputStreamReader isr = new InputStreamReader(bis, "UTF-8");
        SimpleCharStream jSt = new SimpleCharStream(isr);
        return new CastorQLParserTokenManager(jSt);
    }

    // --------------------------------------------------------------------------

}
