/**
 * Copyright (C) 2012 cogroo <cogroo@cogroo.org>
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *         http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.cogroo.tools.checker.rules.util;

import java.io.IOException;
import java.io.InputStream;
import java.util.Properties;

import org.cogroo.tools.checker.rules.applier.RulesTreesBuilder;


/**
 * This class holds common properties to the rules applier classes.
 * 
 * @author Marcelo Suzumura
 * @version $Id: RulesProperties.java 400 2007-04-19 02:36:40Z msuzumura $
 */
public class RulesProperties {

	/**
	 * This resource bundle holds various properties used by the rules applier.
	 */
	//public static final ResourceBundle RESOURCES = ResourceBundle.getBundle("org.cogroo.tools.checker.rules.rules");

	public static String rootFolder = System.getProperty("user.dir") + "/";
	
	/**
	 * Types of trees that will be generated by the {@link RulesTreesBuilder} class, based in the schema and
	 * the rules.
	 */
	public static final int NUMBER_OF_TREES = 3;

	/**
	 * Index of the tree that matches rules in the tokens level.
	 */
	public static final int GENERAL_TREE = 0;

	/**
	 * Index of the tree that matches rules in the chunks level.
	 */
	public static final int PHRASE_LOCAL_TREE = 1;

	/**
	 * Index of the tree that matches rules between the subject and verb.
	 */
	public static final int SUBJECT_VERB_TREE = 2;
	
	//private static final ResourceBundle PROPERTIES = ResourceBundle.getBundle("org.cogroo.tools.checker.rules.rules");
	private static final Properties PROPERTIES = new Properties();
	
  static {
    try {
      InputStream stream = RulesProperties.class.getResourceAsStream("/rules/properties/rules.properties");
      PROPERTIES.load(stream);
      stream.close();
    } catch (IOException e) {
      throw new RuntimeException("could not load rules.properties!", e);
    }
  }

  public static final boolean APPLY_LOCAL = getBoolean("apply.local");
  
  public static final boolean APPLY_PHRASE_LOCAL = getBoolean("apply.phrase.local");

  public static final boolean APPLY_SUBJECT_VERB = getBoolean("apply.subject.verb");

	public static final String PACKAGE = getString("package");
	
	private static String SCHEMA = getString("schema");
	
	private static boolean READ_FROM_SERIALIZED = getBoolean("read.from.serialized");
	
	public static final boolean REREAD_FROM_SERIALIZED = getBoolean("reread.from.serialized");
		
	public static final String DATA_SOURCE = getString("data.source");
	
	private static boolean REREAD_RULES = getBoolean("reread.rules");
	
	public static final String XML_FILE_ENCODING = getString("xml.file.encoding");
	
	private static String XML_FILE = getStringPathWithRoot("xml.file");
	
	public static final String[] CHECKERS = getCommaSeparatedString("checkers");
	
	/**
	 * This resets all paths!
	 * @param value
	 */
	public static void setRootFolder(String value) {
		rootFolder = value + "/";
		
		setRulesFile(getStringPathWithRoot("xml.file"));
		setSchemaFile(getStringPathWithRoot("schema"));
		//setRulesFile(prependWithRoot(getRulesFile()));
		//setSchemaFile(prependWithRoot(getSchemaFile()));
		
		//setSerializedTreesFile(getStringPathWithRoot("serialized.trees.file"));
		
	}
	
	private static String[] getCommaSeparatedString(String string) {
		String[] prop = getString(string).split(",");
		return prop;
	}

	private static boolean getBoolean(String key) {
		return Boolean.parseBoolean(PROPERTIES.getProperty(key));
	}

	private static String getString(String key) {
		return PROPERTIES.getProperty(key);
	}
	
	private static String getStringPathWithRoot(String key) {
		return prependWithRoot(getString(key));
	}
	
	private static String prependWithRoot(String value) {
		if(rootFolder == null)
			rootFolder = System.getProperty("user.dir") + "/";
		return rootFolder + value;
	}

	/**
	 * Private constructor which prevents this class instantiation.
	 */
	private RulesProperties() {
		// Prevents instantiation.
	}
	
	public static void setRulesFile(String rulesFile) {
		XML_FILE = rulesFile;
	}
	
	public static void setSchemaFile(String schemaFile) {
		SCHEMA = schemaFile;
	}

	public static String getRulesFile() {
		return XML_FILE;
	}
	
	public static String getSchemaFile() {
		return SCHEMA;
	}

	public static String getSerializedTreesFile() {
		return getRulesFile() + ".tree";
	}

	public static void setRereadRules(boolean rereadRules) {
		REREAD_RULES = rereadRules;
	}

	public static boolean isRereadRules() {
		return REREAD_RULES;
	}

	public static void setReadFromSerialized(boolean readFromSerialized) {
		READ_FROM_SERIALIZED = readFromSerialized;
	}

	public static boolean isReadFromSerialized() {
		return READ_FROM_SERIALIZED;
	}

}
