/*
 * Decompiled with CFR 0.152.
 */
package com.netflix.conductor.s3.storage;

import com.amazonaws.HttpMethod;
import com.amazonaws.SdkClientException;
import com.amazonaws.services.s3.AmazonS3;
import com.amazonaws.services.s3.model.GeneratePresignedUrlRequest;
import com.amazonaws.services.s3.model.GetObjectRequest;
import com.amazonaws.services.s3.model.ObjectMetadata;
import com.amazonaws.services.s3.model.PutObjectRequest;
import com.amazonaws.services.s3.model.S3Object;
import com.netflix.conductor.common.run.ExternalStorageLocation;
import com.netflix.conductor.common.utils.ExternalPayloadStorage;
import com.netflix.conductor.core.exception.NonTransientException;
import com.netflix.conductor.core.exception.TransientException;
import com.netflix.conductor.core.utils.IDGenerator;
import com.netflix.conductor.s3.config.S3Properties;
import java.io.InputStream;
import java.net.URISyntaxException;
import java.util.Date;
import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class S3PayloadStorage
implements ExternalPayloadStorage {
    private static final Logger LOGGER = LoggerFactory.getLogger(S3PayloadStorage.class);
    private static final String CONTENT_TYPE = "application/json";
    private final IDGenerator idGenerator;
    private final AmazonS3 s3Client;
    private final String bucketName;
    private final long expirationSec;

    public S3PayloadStorage(IDGenerator idGenerator, S3Properties properties, AmazonS3 s3Client) {
        this.idGenerator = idGenerator;
        this.s3Client = s3Client;
        this.bucketName = properties.getBucketName();
        this.expirationSec = properties.getSignedUrlExpirationDuration().getSeconds();
    }

    public ExternalStorageLocation getLocation(ExternalPayloadStorage.Operation operation, ExternalPayloadStorage.PayloadType payloadType, String path) {
        try {
            ExternalStorageLocation externalStorageLocation = new ExternalStorageLocation();
            Date expiration = new Date();
            long expTimeMillis = expiration.getTime() + 1000L * this.expirationSec;
            expiration.setTime(expTimeMillis);
            HttpMethod httpMethod = HttpMethod.GET;
            if (operation == ExternalPayloadStorage.Operation.WRITE) {
                httpMethod = HttpMethod.PUT;
            }
            String objectKey = StringUtils.isNotBlank((CharSequence)path) ? path : this.getObjectKey(payloadType);
            externalStorageLocation.setPath(objectKey);
            GeneratePresignedUrlRequest generatePresignedUrlRequest = new GeneratePresignedUrlRequest(this.bucketName, objectKey).withMethod(httpMethod).withExpiration(expiration);
            externalStorageLocation.setUri(this.s3Client.generatePresignedUrl(generatePresignedUrlRequest).toURI().toASCIIString());
            return externalStorageLocation;
        }
        catch (SdkClientException e) {
            String msg = String.format("Error communicating with S3 - operation:%s, payloadType: %s, path: %s", operation, payloadType, path);
            LOGGER.error(msg, (Throwable)e);
            throw new TransientException(msg, (Throwable)e);
        }
        catch (URISyntaxException e) {
            String msg = "Invalid URI Syntax";
            LOGGER.error(msg, (Throwable)e);
            throw new NonTransientException(msg, (Throwable)e);
        }
    }

    public void upload(String path, InputStream payload, long payloadSize) {
        try {
            ObjectMetadata objectMetadata = new ObjectMetadata();
            objectMetadata.setContentType(CONTENT_TYPE);
            objectMetadata.setContentLength(payloadSize);
            PutObjectRequest request = new PutObjectRequest(this.bucketName, path, payload, objectMetadata);
            this.s3Client.putObject(request);
        }
        catch (SdkClientException e) {
            String msg = String.format("Error uploading to S3 - path:%s, payloadSize: %d", path, payloadSize);
            LOGGER.error(msg, (Throwable)e);
            throw new TransientException(msg, (Throwable)e);
        }
    }

    public InputStream download(String path) {
        try {
            S3Object s3Object = this.s3Client.getObject(new GetObjectRequest(this.bucketName, path));
            return s3Object.getObjectContent();
        }
        catch (SdkClientException e) {
            String msg = String.format("Error downloading from S3 - path:%s", path);
            LOGGER.error(msg, (Throwable)e);
            throw new TransientException(msg, (Throwable)e);
        }
    }

    private String getObjectKey(ExternalPayloadStorage.PayloadType payloadType) {
        StringBuilder stringBuilder = new StringBuilder();
        switch (payloadType) {
            case WORKFLOW_INPUT: {
                stringBuilder.append("workflow/input/");
                break;
            }
            case WORKFLOW_OUTPUT: {
                stringBuilder.append("workflow/output/");
                break;
            }
            case TASK_INPUT: {
                stringBuilder.append("task/input/");
                break;
            }
            case TASK_OUTPUT: {
                stringBuilder.append("task/output/");
            }
        }
        stringBuilder.append(this.idGenerator.generate()).append(".json");
        return stringBuilder.toString();
    }
}

