/*
 * Decompiled with CFR 0.152.
 */
package com.netflix.conductor.s3.storage;

import com.netflix.conductor.common.run.ExternalStorageLocation;
import com.netflix.conductor.common.utils.ExternalPayloadStorage;
import com.netflix.conductor.core.exception.NonTransientException;
import com.netflix.conductor.core.exception.TransientException;
import com.netflix.conductor.core.utils.IDGenerator;
import com.netflix.conductor.s3.config.S3Properties;
import java.io.InputStream;
import java.time.Duration;
import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import software.amazon.awssdk.core.exception.SdkException;
import software.amazon.awssdk.core.sync.RequestBody;
import software.amazon.awssdk.services.s3.S3Client;
import software.amazon.awssdk.services.s3.model.GetObjectRequest;
import software.amazon.awssdk.services.s3.model.PutObjectRequest;
import software.amazon.awssdk.services.s3.presigner.S3Presigner;
import software.amazon.awssdk.services.s3.presigner.model.GetObjectPresignRequest;
import software.amazon.awssdk.services.s3.presigner.model.PutObjectPresignRequest;

public class S3PayloadStorage
implements ExternalPayloadStorage {
    private static final Logger LOGGER = LoggerFactory.getLogger(S3PayloadStorage.class);
    private static final String CONTENT_TYPE = "application/json";
    private final IDGenerator idGenerator;
    private final S3Client s3Client;
    private final S3Presigner s3Presigner;
    private final String bucketName;
    private final long expirationSec;

    public S3PayloadStorage(IDGenerator idGenerator, S3Properties properties, S3Client s3Client, S3Presigner s3Presigner) {
        this.idGenerator = idGenerator;
        this.s3Client = s3Client;
        this.s3Presigner = s3Presigner;
        this.bucketName = properties.getBucketName();
        this.expirationSec = properties.getSignedUrlExpirationDuration().getSeconds();
    }

    public ExternalStorageLocation getLocation(ExternalPayloadStorage.Operation operation, ExternalPayloadStorage.PayloadType payloadType, String path) {
        try {
            String presignedUrl;
            ExternalStorageLocation externalStorageLocation = new ExternalStorageLocation();
            Duration signatureDuration = Duration.ofSeconds(this.expirationSec);
            String objectKey = StringUtils.isNotBlank((CharSequence)path) ? path : this.getObjectKey(payloadType);
            externalStorageLocation.setPath(objectKey);
            if (operation == ExternalPayloadStorage.Operation.WRITE) {
                PutObjectRequest putObjectRequest = (PutObjectRequest)PutObjectRequest.builder().bucket(this.bucketName).key(objectKey).contentType(CONTENT_TYPE).build();
                PutObjectPresignRequest presignRequest = PutObjectPresignRequest.builder().signatureDuration(signatureDuration).putObjectRequest(putObjectRequest).build();
                presignedUrl = this.s3Presigner.presignPutObject(presignRequest).url().toString();
            } else {
                GetObjectRequest getObjectRequest = (GetObjectRequest)GetObjectRequest.builder().bucket(this.bucketName).key(objectKey).build();
                GetObjectPresignRequest presignRequest = GetObjectPresignRequest.builder().signatureDuration(signatureDuration).getObjectRequest(getObjectRequest).build();
                presignedUrl = this.s3Presigner.presignGetObject(presignRequest).url().toString();
            }
            externalStorageLocation.setUri(presignedUrl);
            return externalStorageLocation;
        }
        catch (SdkException e) {
            String msg = String.format("Error communicating with S3 - operation:%s, payloadType: %s, path: %s", operation, payloadType, path);
            LOGGER.error(msg, (Throwable)e);
            throw new TransientException(msg, (Throwable)e);
        }
        catch (Exception e) {
            String msg = "Error generating presigned URL";
            LOGGER.error(msg, (Throwable)e);
            throw new NonTransientException(msg, (Throwable)e);
        }
    }

    public void upload(String path, InputStream payload, long payloadSize) {
        try {
            PutObjectRequest request = (PutObjectRequest)PutObjectRequest.builder().bucket(this.bucketName).key(path).contentType(CONTENT_TYPE).contentLength(Long.valueOf(payloadSize)).build();
            this.s3Client.putObject(request, RequestBody.fromInputStream((InputStream)payload, (long)payloadSize));
        }
        catch (SdkException e) {
            String msg = String.format("Error uploading to S3 - path:%s, payloadSize: %d", path, payloadSize);
            LOGGER.error(msg, (Throwable)e);
            throw new TransientException(msg, (Throwable)e);
        }
    }

    public InputStream download(String path) {
        try {
            GetObjectRequest request = (GetObjectRequest)GetObjectRequest.builder().bucket(this.bucketName).key(path).build();
            return this.s3Client.getObject(request);
        }
        catch (SdkException e) {
            String msg = String.format("Error downloading from S3 - path:%s", path);
            LOGGER.error(msg, (Throwable)e);
            throw new TransientException(msg, (Throwable)e);
        }
    }

    private String getObjectKey(ExternalPayloadStorage.PayloadType payloadType) {
        StringBuilder stringBuilder = new StringBuilder();
        switch (payloadType) {
            case WORKFLOW_INPUT: {
                stringBuilder.append("workflow/input/");
                break;
            }
            case WORKFLOW_OUTPUT: {
                stringBuilder.append("workflow/output/");
                break;
            }
            case TASK_INPUT: {
                stringBuilder.append("task/input/");
                break;
            }
            case TASK_OUTPUT: {
                stringBuilder.append("task/output/");
            }
        }
        stringBuilder.append(this.idGenerator.generate()).append(".json");
        return stringBuilder.toString();
    }
}

