/*
 * Copyright 2008 the original author or authors.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.constretto.spring;

import org.constretto.ConstrettoConfiguration;
import org.constretto.annotation.Configuration;
import org.constretto.spring.annotation.Environment;
import org.constretto.spring.resolver.AssemblyContextResolver;
import org.springframework.beans.BeanInstantiationException;
import org.springframework.beans.BeansException;
import org.springframework.beans.factory.config.BeanPostProcessor;
import org.springframework.beans.factory.config.InstantiationAwareBeanPostProcessorAdapter;

import java.lang.reflect.Field;

/**
 * {@link BeanPostProcessor} implementation that autowires annotated fields annotated with the &#064;Configuration or
 * &#064;Environment annotations.
 * <p/>
 * <p/>
 * Fields are injected right after construction of a bean, before any config methods are invoked. Such a config field
 * does not have to be public.
 * <p/>
 * <p/>
 * If a &#064;Configuration element have the required flag set, and no value could be assosiated with the given key a
 * {@link BeanInstantiationException} is thrown invalidating the entire context.
 *
 * @author <a href="mailto:kaare.nilsen@gmail.com">Kaare Nilsen</a>
 * @see Configuration
 * @see org.constretto.spring.annotation.Environment
 */
public class ConfigurationAnnotationConfigurer extends InstantiationAwareBeanPostProcessorAdapter {
    private ConstrettoConfiguration configuration;
    private AssemblyContextResolver assemblyContextResolver;

    public ConfigurationAnnotationConfigurer(ConstrettoConfiguration configuration,
                                             AssemblyContextResolver assemblyContextResolver) {
        this.configuration = configuration;
        this.assemblyContextResolver = assemblyContextResolver;
    }

    @Override
    public boolean postProcessAfterInstantiation(Object bean, String beanName) throws BeansException {
        injectConfiguration(bean);
        injectEnvironment(bean);
        return true;
    }

    public Object postProcessAfterInitialization(Object bean, String beanName) throws BeansException {
        return bean;
    }

    private void injectEnvironment(final Object bean) {
        try {
            Field[] fields = bean.getClass().getDeclaredFields();
            for (Field field : fields) {
                if (field.isAnnotationPresent(Environment.class)) {
                    field.setAccessible(true);
                    field.set(bean, assemblyContextResolver.getAssemblyContext());
                }
            }
        } catch (IllegalAccessException e) {
            throw new BeanInstantiationException(bean.getClass(), "Could not inject Environment on spring bean");
        }
    }

    private void injectConfiguration(final Object bean) {
        configuration.on(bean);
    }
}
