
/*
 * Copyright (c) 2020 coodex.org (jujus.shen@126.com)
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.coodex.id;

import org.coodex.config.Config;
import org.coodex.functional.Supplier;
import org.coodex.util.Clock;
import org.coodex.util.Common;
import org.coodex.util.Profile;
import org.coodex.util.Singleton;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

/**
 * Twitter_Snowflake<br>
 * SnowFlake的结构如下(每部分用-分开):<br>
 * 0 - 0000000000 0000000000 0000000000 0000000000 0 - 00000 - 00000 - 000000000000 <br>
 * 1位标识，由于long基本类型在Java中是带符号的，最高位是符号位，正数是0，负数是1，所以id一般是正数，最高位是0<br>
 * 41位时间截(毫秒级)，注意，41位时间截不是存储当前时间的时间截，而是存储时间截的差值（当前时间截 - 开始时间截)
 * 得到的值），这里的的开始时间截，一般是我们的id生成器开始使用的时间，由我们程序来指定的（如下下面程序IdWorker类的startTime属性）。41位的时间截，可以使用69年，年T = (1L << 41) / (1000L * 60 * 60 * 24 * 365) = 69<br>
 * 10位的数据机器位，可以部署在1024个节点，包括5位dataCenterId和5位workerId<br>
 * 12位序列，毫秒内的计数，12位的计数顺序号支持每个节点每毫秒(同一机器，同一时间截)产生4096个ID序号<br>
 * 加起来刚好64位，为一个Long型。<br>
 * SnowFlake的优点是，整体上按照时间自增排序，并且整个分布式系统内不会产生ID碰撞(由数据中心ID和机器ID作区分)，并且效率较高，经测试，SnowFlake每秒能够产生26万ID左右。
 */
public class SnowflakeIdWorker {

    private static final Logger log = LoggerFactory.getLogger(SnowflakeIdWorker.class);


    // ==============================Fields===========================================
    static final Singleton<SnowflakeIdWorker> snowflakeIdWorkerSingleton
            = Singleton.with(
            new Supplier<SnowflakeIdWorker>() {
                @Override
                public SnowflakeIdWorker get() {
                    SnowflakeIdWorker snowflakeIdWorker;
                    int machineId = Config.getValue("snowflake.machineId", -1);

                    if (machineId != -1) {
                        snowflakeIdWorker = new SnowflakeIdWorker(machineId);
                    } else {
                        int workerId = Config.getValue("snowflake.workerId", -1);
                        int dataCenterId = Config.getValue("snowflake.dataCenterId", -1);
                        if (workerId == -1 && dataCenterId == -1) {
                            machineId = Profile.get("idWorker").getInt("machineId", -1);
                            if (machineId == -1) {
                                log.warn("snowflake parameters[machineId, workerId, dataCenterId] not set. use default value.");
                                machineId = 0;
                            }
                            snowflakeIdWorker = new SnowflakeIdWorker(machineId);
                        } else {
                            snowflakeIdWorker = new SnowflakeIdWorker(workerId, dataCenterId);
                        }
                    }
                    return snowflakeIdWorker;
                }
            }
//            () -> {
//                SnowflakeIdWorker snowflakeIdWorker;
//                int machineId = Config.getValue("snowflake.machineId", -1);
//
//                if (machineId != -1) {
//                    snowflakeIdWorker = new SnowflakeIdWorker(machineId);
//                } else {
//                    int workerId = Config.getValue("snowflake.workerId", -1);
//                    int dataCenterId = Config.getValue("snowflake.dataCenterId", -1);
//                    if (workerId == -1 && dataCenterId == -1) {
//                        machineId = Profile.get("idWorker").getInt("machineId", -1);
//                        if (machineId == -1) {
//                            log.warn("snowflake parameters[machineId, workerId, dataCenterId] not set. use default value.");
//                            machineId = 0;
//                        }
//                        snowflakeIdWorker = new SnowflakeIdWorker(machineId);
//                    } else {
//                        snowflakeIdWorker = new SnowflakeIdWorker(workerId, dataCenterId);
//                    }
//                }
//                return snowflakeIdWorker;
//            }
//                    new SnowflakeIdWorker(
//                    Config.getValue(
//                            "snowflake.machineId",
//                            () -> Profile.get("idWorker").getInt("machineId", 0)
//                    )
//            )
    );
    /**
     * 开始时间截 (2020-01-01)
     */
    private final long twepoch = Common.calendar(2020).getTimeInMillis();
    /**
     * 机器id所占的位数
     */
    private final long workerIdBits = 5L;
    /**
     * 数据标识id所占的位数
     */
    private final long dataCenterIdBits = 5L;
    /**
     * 支持的最大机器id，结果是31 (这个移位算法可以很快的计算出几位二进制数所能表示的最大十进制数)
     */
    @SuppressWarnings("FieldCanBeLocal")
    private final long maxWorkerId = ~(-1L << workerIdBits);
    /**
     * 支持的最大数据标识id，结果是31
     */
    @SuppressWarnings("FieldCanBeLocal")
    private final long maxDataCenterId = ~(-1L << dataCenterIdBits);
    /**
     * 序列在id中占的位数
     */
    private final long sequenceBits = 12L;
    /**
     * 机器ID向左移12位
     */
    @SuppressWarnings("FieldCanBeLocal")
    private final long workerIdShift = sequenceBits;
    /**
     * 数据标识id向左移17位(12+5)
     */
    @SuppressWarnings("FieldCanBeLocal")
    private final long dataCenterIdShift = sequenceBits + workerIdBits;
    /**
     * 时间截向左移22位(5+5+12)
     */
    @SuppressWarnings("FieldCanBeLocal")
    private final long timestampLeftShift = sequenceBits + workerIdBits + dataCenterIdBits;
    /**
     * 生成序列的掩码，这里为4095 (0b111111111111=0xfff=4095)
     */
    @SuppressWarnings("FieldCanBeLocal")
    private final long sequenceMask = ~(-1L << sequenceBits);
    /**
     * 工作机器ID(0~31)
     */
    private final long workerId;
    /**
     * 数据中心ID(0~31)
     */
    private final long dataCenterId;
    /**
     * 毫秒内序列(0~4095)
     */
    private long sequence = 0L;

    //==============================Constructors=====================================
    /**
     * 上次生成ID的时间截
     */
    private long lastTimestamp = -1L;

    /**
     * @param machineId 机器id(0~1023)
     */
    public SnowflakeIdWorker(int machineId) {
        this(machineId & 0x1F, machineId >> 5);
    }
    // ==============================Methods==========================================


    /**
     * 构造函数
     *
     * @param workerId     工作ID (0~31)
     * @param dataCenterId 数据中心ID (0~31)
     */
    public SnowflakeIdWorker(long workerId, long dataCenterId) {
        if (workerId > maxWorkerId || workerId < 0) {
            throw new IllegalArgumentException(String.format("worker Id can't be greater than %d or less than 0", maxWorkerId));
        }
        if (dataCenterId > maxDataCenterId || dataCenterId < 0) {
            throw new IllegalArgumentException(String.format("datacenter Id can't be greater than %d or less than 0", maxDataCenterId));
        }
        this.workerId = workerId;
        this.dataCenterId = dataCenterId;
    }

//    /**
//     * 测试
//     */
//    public static void main(String[] args) {
//        SnowflakeIdWorker idWorker = new SnowflakeIdWorker(0, 0);
//        for (int i = 0; i < 1000; i++) {
//            long id = idWorker.nextId();
//            System.out.println(Long.toBinaryString(id));
//            System.out.println(id);
//            System.out.println(String.format("%016x", id));
//            System.out.println(Base58.encode(long2Bytes(id)));
//        }
//    }


//    // 大小写敏感场景使用
//    public static String getBase58Id() {
//        return Base58.encode(long2Bytes(getId()));
//    }
//
//    // 大小写不敏感场景使用
//    public static String getBase16Id() {
//        return String.format("%x", getId());
//    }

    // 直接用数值
    public static long getId() {
        return snowflakeIdWorkerSingleton.get().nextId();
    }

    //==============================Test=============================================

    public static Info parse(long snowflakeId) {
        return new Info(snowflakeId);
    }

    /**
     * 获得下一个ID (该方法是线程安全的)
     *
     * @return SnowflakeId
     */
    public synchronized long nextId() {
        long timestamp = timeGen();

        //如果当前时间小于上一次ID生成的时间戳，说明系统时钟回退过这个时候应当抛出异常
        if (timestamp < lastTimestamp) {
            throw new RuntimeException(
                    String.format("Clock moved backwards.  Refusing to generate id for %d milliseconds", lastTimestamp - timestamp));
        }

        //如果是同一时间生成的，则进行毫秒内序列
        if (lastTimestamp == timestamp) {
            sequence = (sequence + 1) & sequenceMask;
            //毫秒内序列溢出
            if (sequence == 0) {
                //阻塞到下一个毫秒,获得新的时间戳
                timestamp = tilNextMillis(lastTimestamp);
            }
        }
        //时间戳改变，毫秒内序列重置
        else {
            sequence = 0L;
        }

        //上次生成ID的时间截
        lastTimestamp = timestamp;

        //移位并通过或运算拼到一起组成64位的ID
        return ((timestamp - twepoch) << timestampLeftShift) //
                | (dataCenterId << dataCenterIdShift) //
                | (workerId << workerIdShift) //
                | sequence;
    }

    /**
     * 阻塞到下一个毫秒，直到获得新的时间戳
     *
     * @param lastTimestamp 上次生成ID的时间截
     * @return 当前时间戳
     */
    protected long tilNextMillis(long lastTimestamp) {
        long timestamp = timeGen();
        while (timestamp <= lastTimestamp) {
            timestamp = timeGen();
        }
        return timestamp;
    }

    /**
     * 返回以毫秒为单位的当前时间
     *
     * @return 当前时间(毫秒)
     */
    protected long timeGen() {
        return Clock.currentTimeMillis();
    }

    public static class Info {
        public final int workerId;
        public final int dataCenterId;
        public final int seq;
        public final long timestamp;

        Info(long id) {
            seq = (int) (id & 0xFFF);
            id >>= 12;
            workerId = (int) (id & 0x1F);
            id >>= 5;
            dataCenterId = (int) (id & 0x1F);
            id >>= 5;
            timestamp = id & ~(-1L << 41);
        }

        @Override
        public String toString() {
            return "Info{" +
                    "workerId=" + workerId +
                    ", dataCenterId=" + dataCenterId +
                    ", seq=" + seq +
                    ", timestamp=" + timestamp +
                    '}';
        }
    }

}