/*
 * Copyright (c) 2018 coodex.org (jujus.shen@126.com)
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.coodex.util;

import org.coodex.functional.BiConsumer;
import org.coodex.functional.Consumer;
import org.coodex.functional.Function;
import org.coodex.functional.Supplier;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.lang.annotation.Annotation;
import java.lang.reflect.*;
import java.net.URL;
import java.util.*;

import static org.coodex.util.Common.cast;

/**
 * @author Davidoff
 */
public class ReflectHelper {


    public static final Function<Class<?>, Boolean> NOT_NULL = new Function<Class<?>, Boolean>() {
        @Override
        public Boolean apply(Class<?> aClass) {
            return aClass != null;
        }
    }/*Objects::nonNull*/;
    @SuppressWarnings("unused")
    public static final Function<Class<?>, Boolean> ALL_OBjECT = new Function<Class<?>, Boolean>() {
        @Override
        public Boolean apply(Class<?> c) {
            return c != null && c != Object.class;
        }
    }/*c -> c != null && c != Object.class */;
    @SuppressWarnings("unused")
    public static final Function<Class<?>, Boolean> ALL_OBJECT_EXCEPT_JAVA_SDK = new Function<Class<?>, Boolean>() {
        @Override
        public Boolean apply(Class<?> c) {
            return c != null && !c.getName().startsWith("java");
        }
    }/*c -> c != null && !c.getName().startsWith("java")*/;
    private static final Logger log = LoggerFactory.getLogger(ReflectHelper.class);

    private ReflectHelper() {
    }

    public static boolean classExists(String className) {
        try {
            Class.forName(className);
            return true;
        } catch (ClassNotFoundException e) {
            return false;
        }
    }

    public static <T extends Annotation> T getAnnotation(Class<T> annotationClass, AnnotatedElement element, Set<AnnotatedElement> checked) {
        if (checked.contains(element)) {
            return null;
        }
        checked.add(element);
        T t = element.getAnnotation(annotationClass);
        if (t != null) {
            return t;
        }
        for (Annotation annotation : element.getAnnotations()) {
            t = getAnnotation(annotationClass, annotation.annotationType(), checked);
            if (t != null) {
                return t;
            }
        }
        return null;
    }

    public static <T extends Annotation> T getAnnotation(Class<T> annotationClass, AnnotatedElement... elements) {
        Set<AnnotatedElement> checked = new HashSet<>();
        for (AnnotatedElement element : elements) {
            T t = getAnnotation(annotationClass, element, checked);
            if (t != null) {
                return t;
            }
        }
        return null;
    }

    public static String getParameterName(Object executable, int index, String prefix) {
//        if (executable instanceof Executable) {
        if (executable instanceof Method) {
            String parameterName = getParameterName((Method) executable, index);
            return parameterName == null ? (prefix + index) : parameterName;
        } else if (executable instanceof Constructor) {
            String parameterName = getParameterName((Constructor<?>) executable, index);
            return parameterName == null ? (prefix + index) : parameterName;
        } else {
            throw new IllegalArgumentException("none Executable object: " + executable);
        }
    }

    private static Method getMethod(Class<?> objClass, String methodName) throws NoSuchMethodException {
        for (Method method : objClass.getDeclaredMethods()) {
            if (methodName.equals(method.getName())) {
                return method;
            }
        }
        throw new NoSuchMethodException();
    }

    private static Object invoke(Object obj, String methodName, Object... args) throws ReflectiveOperationException {
        Field overrideField = AccessibleObject.class.getDeclaredField("override");
        overrideField.setAccessible(true);
        Method targetMethod = getMethod(obj.getClass(), methodName);
        overrideField.set(targetMethod, true);
        return targetMethod.invoke(obj, args);
    }

    /**
     * 获取lambda表达式实例的方法签名，方法来源：https://zhuanlan.zhihu.com/p/151438084
     * <p>
     * 原理，在lambda实例的class(isSynthetic)中找到对应的方法，因为lambda表达式是单一方法的实例，所以能够找到的应该是有且仅有的一个方法
     *
     * @param lambda lambda instance
     * @return lambda表达式实例的方法签名
     * @throws ReflectiveOperationException exception
     */
    @SuppressWarnings("unused")
    public static Method getLambdaMethod(Object lambda) throws ReflectiveOperationException {
        Class<?> lambdaClass = lambda.getClass();
        if (!lambdaClass.isSynthetic()) {
            throw new IllegalArgumentException("not lambda instance: " + lambdaClass);
        }
        Object constantPool = invoke(lambdaClass, "getConstantPool");
        for (int i = (int) invoke(constantPool, "getSize") - 1; i >= 0; --i) {
            try {
                Object member = invoke(constantPool, "getMethodAt", i);
                if (member instanceof Method && ((Method) member).isSynthetic() && ((Method) member).getDeclaringClass() != Object.class) {
                    return (Method) member;
                }
            } catch (Exception ignored) {// NOSONAR
                // ignored
            }
        }
        throw new NoSuchMethodException();
    }

    public static String getParameterName(Method method, int index) {
        String s = getParameterNameByAnnotation(method.getParameterAnnotations(), index);
        return s == null ? getParameterNameByJava8(method, index) : s;
    }

    private static String getParameterNameByAnnotation(Annotation[][] annotations, int index) {
        if (annotations == null || annotations.length < index) {
            return null;
        }

        for (Annotation annotation : annotations[index]) {
            if (annotation instanceof Parameter) {
                return ((Parameter) annotation).value();
            }
        }
        return null;
    }

//    public static String getParameterName(Method executable, int index) {
//        String s = getParameterNameByAnnotation(executable.getParameterAnnotations(), index);
//
//        return s == null ? getParameterNameByJava8(executable, index) : s;
//    }

    public static String getParameterName(Constructor<?> executable, int index) {
        String s = getParameterNameByAnnotation(executable.getParameterAnnotations(), index);

        return s == null ? getParameterNameByJava8(executable, index) : s;
    }

//    private static String getParameterNameByJava8(Executable executable, int index) {
//        return executable.getParameters()[index].getName();
//    }

    private static String getParameterNameByJava8(Method executable, int index) {
        try {
            return executable.getParameters()[index].getName();
        } catch (Throwable th) {
            return "arg" + index;
        }
    }

    private static String getParameterNameByJava8(Constructor<?> executable, int index) {
        try {
            return executable.getParameters()[index].getName();
        } catch (Throwable th) {
            return "arg" + index;
        }
    }

    public static Field[] getAllDeclaredFields(Class<?> clz) {
        return getAllDeclaredFields(clz, null);
    }

    public static Field[] getAllDeclaredFields(Class<?> clz,
                                               Function<Class<?>, Boolean> decision) {
        if (clz == null) {
            throw new NullPointerException("class is NULL");
        }
        if (decision == null) {
            decision = NOT_NULL;
        }
        Map<String, Field> fields = new HashMap<>();
        Class<?> clazz = clz;
        while (decision.apply(clazz)) {
            Field[] declaredFields = clazz.getDeclaredFields();
            for (Field field : declaredFields) {
                if (!fields.containsKey(field.getName())) {
                    fields.put(field.getName(), field);
                }
            }
            clazz = clazz.getSuperclass();
        }
        return fields.values().toArray(new Field[0]);
    }

    public static Object invoke(Object obj, Method method, Object[] args)
            throws IllegalAccessException, IllegalArgumentException,
            InvocationTargetException, NoSuchMethodException, SecurityException {
        if (obj == null) {
            throw new NullPointerException("invoke target object is NULL.");
        }

        if (method.getDeclaringClass().isAssignableFrom(obj.getClass())) {
            return method.invoke(obj, args);
        } else {
            return obj.getClass()
                    .getMethod(method.getName(), method.getParameterTypes())
                    .invoke(obj, args);
        }
    }

    private static String resourceToClassName(String resourceName) {
        if (resourceName.endsWith(".class")) {
            return resourceName.substring(0, resourceName.length() - 6).replace('/', '.');
        }
        return null;
    }

    private static String[] packageToPath(String[] packages) {
        if (packages == null || packages.length == 0) {
            return new String[0];
        }
        String[] paths = new String[packages.length];
        int i = 0;
        for (String p : packages) {
            paths[i++] = p == null ? "" : p.replace('.', '/');
        }
        return paths;
    }

    public static void foreachClass(final Consumer<Class<?>> processor, final Function<String, Boolean> filter, String... packages) {
        if (processor == null) {
            return;
        }
        ResourceScanner.newBuilder(
                        new BiConsumer<URL, String>() {
                            @Override
                            public void accept(URL url, String resourceName) {
                                String className = resourceToClassName(resourceName);
                                try {
                                    processor.accept(Class.forName(className));
                                } catch (ClassNotFoundException e) {
                                    log.warn("load class fail. {}, {}", className, e.getLocalizedMessage());
                                }
                            }
                        }
//                (resource, resourceName) -> {
//                    String className = resourceToClassName(resourceName);
//                    try {
//                        processor.accept(Class.forName(className));
//                    } catch (ClassNotFoundException e) {
//                        log.warn("load class fail. {}, {}", className, e.getLocalizedMessage());
//                    }
//                }
                ).filter(new Function<String, Boolean>() {
                    @Override
                    public Boolean apply(String resourceName) {
                        String className = resourceToClassName(resourceName);
                        return className != null && filter.apply(className);
                    }
                }).build()
                .scan(packageToPath(packages)
                );
    }

    @SuppressWarnings("unused")
    public static <T> T throwExceptionObject(Class<T> interfaceClass, final Supplier<Throwable> supplier) {
        return cast(Proxy.newProxyInstance(interfaceClass.getClassLoader(), new Class<?>[]{interfaceClass},
                new InvocationHandler() {
                    @Override
                    public Object invoke(Object proxy, Method method, Object[] args) throws Throwable {
                        throw supplier.get();
                    }
                }));
    }

    @SuppressWarnings("unused")
    public static <T> T throwExceptionObject(Class<T> interfaceClass, final Function<Method, Throwable> function) {
        return cast(Proxy.newProxyInstance(interfaceClass.getClassLoader(), new Class<?>[]{interfaceClass},
                new InvocationHandler() {
                    @Override
                    public Object invoke(Object proxy, Method method, Object[] args) throws Throwable {
                        throw function.apply(method);
                    }
                }));
    }

    public static String typeToCodeStr(Type type) {
        StringBuilder builder = new StringBuilder();
        if (type instanceof ParameterizedType) {
            builder.append(typeToCodeStr(((ParameterizedType) type).getRawType())).append("<");
            int l = ((ParameterizedType) type).getActualTypeArguments().length;
            for (int i = 0; i < l; i++) {
                if (i > 0) {
                    builder.append(", ");
                }
                builder.append(typeToCodeStr(((ParameterizedType) type).getActualTypeArguments()[i]));
            }
            builder.append(">");
        } else if (type instanceof Class) {
            if (((Class<?>) type).isArray()) {
                return typeToCodeStr(((Class<?>) type).getComponentType()) + "[]";
            } else {
                return ((Class<?>) type).getName();
            }
        } else if (type instanceof TypeVariable) {
            return ((TypeVariable<?>) type).getName();
        } else if (type instanceof GenericArrayType) {
            return typeToCodeStr(((GenericArrayType) type).getGenericComponentType()) + "[]";
        }
        return builder.toString();
    }

    private static Object invoke(Method method, Object first, Object[] objects, Object[] args) throws Throwable {
        if (first == null) {
            throw new NullPointerException();
        }
        if (method.getDeclaringClass().isAssignableFrom(first.getClass())) {
            return args == null || args.length == 0 ? method.invoke(first) : method.invoke(first, args);
        }
        for (Object o : objects) {
            if (o == null) {
                throw new NullPointerException();
            }
            if (method.getDeclaringClass().isAssignableFrom(o.getClass())) {
                return args == null || args.length == 0 ? method.invoke(o) : method.invoke(o, args);
            }
        }
        throw new RuntimeException("method not found in all objects: " + method.getName());
    }

    public static Class<?>[] getAllInterfaces(Class<?> clz) {
        Collection<Class<?>> coll = new HashSet<>();
        addInterfaceTo(clz, coll);
        return coll.toArray(new Class<?>[0]);
    }

    private static void addInterfaceTo(Class<?> clz, Collection<Class<?>> coll) {
        if (clz == null) {
            return;
        }
        if (coll.contains(clz)) {
            return;
        }
        if (clz.isInterface()) {
            coll.add(clz);
        }
        addInterfaceTo(clz.getSuperclass(), coll);
        for (Class<?> c : clz.getInterfaces()) {
            addInterfaceTo(c, coll);
        }
    }

    public static boolean isAssignable(Class<?> from, Class<?> to) {
        if (from.isArray() && to.isArray()) {
            return Objects.equals(from, to);
        }
        if (from.isArray() || to.isArray()) {
            return false;
        }
        return to.isAssignableFrom(from);
    }

    public static boolean isMatch(Type instanceType, Type serviceType) {
        boolean match = isMatchForDebug(instanceType, serviceType);
        if (Common.isDebug() && log.isDebugEnabled()) {
            log.debug("match: {}\ninstance: {}\nservice: {} ", match, instanceType, serviceType);
        }
        return match;
    }

    private static boolean isMatchForDebug(Type instanceType, Type serviceType) {
        if (Objects.equals(instanceType, serviceType)) {
            return true;
        }
        Class<?> instanceClass = GenericTypeHelper.typeToClass(instanceType);
        Class<?> serviceClass = GenericTypeHelper.typeToClass(serviceType);

        // 实例类型与服务类型的class不匹配
        if (serviceClass != null && instanceClass != null && !isAssignable(instanceClass, serviceClass)) {
            return false;
        }

        if (serviceClass != null && serviceType instanceof ParameterizedType) {
            ParameterizedType parameterizedServiceType = (ParameterizedType) serviceType;
            for (int i = 0; i < parameterizedServiceType.getActualTypeArguments().length; i++) {
                if (!isMatch(
                        GenericTypeHelper.solveFromType(serviceClass.getTypeParameters()[i], instanceType),
                        parameterizedServiceType.getActualTypeArguments()[i])) {
                    return false;
                }
            }
            return true;
        } else {
            return !(serviceType instanceof GenericArrayType);// 均为泛型数组，则必须完全相同，在第一行已处理
        }
    }

    /**
     * @param o       object
     * @param objects 需要扩展出来的对象，只扩展接口
     * @param <S>     必须是接口
     * @param <T>     extends S
     * @return 扩展后的对象
     */
    @SuppressWarnings("unused")
    public static <S, T extends S> S extendInterface(final T o, final Object... objects) {
        if (o == null) {
            return null;
        }
        if (objects == null || objects.length == 0) {
            return o;
        }
        Set<Class<?>> interfaces = new HashSet<>();
        addInterfaceTo(o.getClass(), interfaces);
        for (Object x : objects) {
            if (x != null) {
                addInterfaceTo(x.getClass(), interfaces);
            }
        }
        if (interfaces.isEmpty()) {
            return o;
        }

        return cast(Proxy.newProxyInstance(Thread.currentThread().getContextClassLoader(), interfaces.toArray(new Class<?>[0]),
                new InvocationHandler() {
                    @Override
                    public Object invoke(Object proxy, Method method, Object[] args) throws Throwable {
                        return ReflectHelper.invoke(method, o, objects, args);
                    }
                }));
    }

    public static class MethodParameter {

        private final Method method;
        private final int index;
        private final String name;
        private final Annotation[] annotations;
        private final Class<?> type;
        private final Type genericType;


        public MethodParameter(Method method, int index) {
            this.method = method;
            this.index = index;

            annotations = method.getParameterAnnotations()[index];
            type = method.getParameterTypes()[index];
            genericType = method.getGenericParameterTypes()[index];

            name = getParameterName(method, index, "p");

        }

        public Class<?> getType() {
            return type;
        }

        public Type getGenericType() {
            return genericType;
        }

        public Method getMethod() {
            return method;
        }

        public int getIndex() {
            return index;
        }

        public String getName() {
            return name;
        }

        public Annotation[] getAnnotations() {
            return annotations;
        }


        public <T> T getAnnotation(Class<T> annotationClass) {
            if (annotationClass == null) {
                throw new IllegalArgumentException("annotationClass is NULL.");
            }
            if (annotations == null) {
                return null;
            }
            for (Annotation annotation : annotations) {
                if (annotationClass.isAssignableFrom(annotation.getClass())) {
                    return Common.cast(annotation);
                }
            }
            return null;
        }
    }

}
