/**
 * COOS - Connected Objects Operating System (www.connectedobjects.org).
 *
 * Copyright (C) 2009 Telenor ASA and Tellu AS. All rights reserved.
 * DO NOT ALTER OR REMOVE COPYRIGHT NOTICES OR THIS FILE HEADER.
 *
 * This library is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as published
 * by the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 *
 * You may also contact one of the following for additional information:
 * Telenor ASA, Snaroyveien 30, N-1331 Fornebu, Norway (www.telenor.no)
 * Tellu AS, Hagalokkveien 13, N-1383 Asker, Norway (www.tellu.no)
 */
package org.coos.javaframe;

import org.coos.actorframe.messages.AFConstants;
import org.coos.javaframe.messages.AFPropertyMsg;
import org.coos.javaframe.messages.ActorMsg;

import java.util.Vector;

/**
 * @author Geir Melby, Tellu AS
 */
public class RoleCS extends CompositeState implements AFConstants {

	protected State idle = new State("idle", this);

	ActorAddress parentAddress;

	public RoleCS(String s) {
		super(s);
	}

	/**
	 * The constructor
	 * 
	 * @param sn
	 *            The name of the state
	 * @param cs
	 *            The enclosing state
	 */
	public RoleCS(String sn, CompositeState cs) {
		super(sn, cs);
	}

	/**
	 * Makes the current statemachine enter idle
	 * 
	 * @param curfsm
	 *            The current statemachine
	 */
	public void enterState(StateMachine curfsm) {
		curfsm.setInitialState(idle);
		idle.enterState(curfsm);
	}

	public ActorAddress getParentAddress() {
		return parentAddress;
	}

	public void setParentAddress(ActorAddress parentAddress) {
		this.parentAddress = parentAddress;
	}

	public void nextState(State st) {
		nextState(st, curfsm);
	}

	/**
	 * Sets the next state, where the new state is a composiste state. The entry
	 * point is defined
	 * 
	 * @param st
	 *            is the new state
	 * @param entryNo
	 *            is the entry number of the composiste state
	 */
	public void nextState(State st, int entryNo) {
		nextState(st, entryNo, curfsm);
	}

	/**
	 * Remain in the same State of the StateMachine. The method is to be used
	 * within the method execTrans(). Notice that entry() of this State will not
	 * be executed. That means performExit() should neither have been called.
	 */
	public void sameState() {
		sameState(curfsm);
	}

	/**
	 * Exit current state. The extit method is run.
	 */
	public void performExit() {
		performExit(curfsm);
	}

	/**
	 * Execute exit() from currentState up to (but not including) this
	 * CompositeState. The method is to be used within the method execTrans() of
	 * RoleCS type. An performExit flag in StateMachine is set to enable check
	 * of inkonsistent use of nextstate and performExit in the transistion. If
	 * this method has been called before during the same transition, the call
	 * should be ignored
	 */
	public void performExit(CompositeState cs) {
		performExit(curfsm, cs);
	}

	/**
	 * Saves the sig into current state machine
	 * 
	 * @param sig
	 *            is the signal to be saved
	 */
	public void save(ActorMsg sig) {
		save(sig, curfsm);
	}

	/**
	 * Start a timer with length as milliseconds and the name of the timer
	 * 
	 * @param duration
	 *            is the length of the timer in milliseconds
	 * @param timerId
	 *            is the timer id, used to identify the timer at the receiving
	 *            time.
	 */
	public void startTimer(long duration, String timerId) {
		curfsm.startTimer(duration, timerId);
	}

	/**
	 * Sends a actor message with signal name and the receiver of the message.
	 * An AFProperty message will be created with no properties. a simple
	 * message with no properties.
	 * 
	 * @param sigName
	 *            is the name of the signal
	 * @param receiver
	 *            is the receiver addres specified as a string on format
	 *            <actorId>:<PortName>@<ActorType. Port name is optional
	 */
	public void sendMessage(String sigName, String receiver) {
		curfsm.sendMessage(sigName, receiver);
	}

	public void stopTimer(String timerId) {
		curfsm.stopTimer(timerId);
	}

	/**
	 * Sends a actor message with signal name and the receiver of the message.
	 * An AFProperty message will be created with no properties. a simple
	 * message with no properties.
	 * 
	 * @param sigName
	 *            is the name of the signal
	 * @param receiver
	 *            is the receiver addres specified as an ActorAddress
	 */

	public void sendMessage(String sigName, ActorAddress receiver) {
		curfsm.sendMessage(sigName, receiver);
	}

	/**
	 * Sends a actor message with signal name and the receiver of the message.
	 * 
	 * @param msg
	 *            is the message
	 * @param portName
	 *            is the name of the port where the signal is send via.
	 *            <actorId>:<PortName>@<ActorType. Port name is optional
	 */
	public void sendMessage(ActorMsg msg, String portName) {
		curfsm.sendMessage(msg, portName);
	}

	/**
	 * Sends a actor message with signal name and the receiver of the message.
	 * An AFProperty message will be created with no properties. a simple
	 * message with no properties.
	 * 
	 * @param msg
	 *            is the message
	 * @param receivers
	 *            is an vector of actor addresses
	 */
	public void sendMessage(ActorMsg msg, Vector receivers) {
		curfsm.sendMessage(msg, receivers);
	}

	/**
	 * Sends a actor message with signal name and a vector of actor addresses.
	 * An AFProperty message will be created with no properties. a simple
	 * message with no properties.
	 * 
	 * @param signaName
	 *            is the name of the message
	 * @param receivers
	 *            is an vector of actor addresses
	 */
	public void sendMessage(String signaName, Vector receivers) {
		curfsm.sendMessage(new AFPropertyMsg(signaName), receivers);
	}

	/**
	 * Sends a actor message. The receiver of the message has to be set before
	 * this method is send
	 * 
	 * @param msg
	 *            is the message <actorId>:<PortName>@<ActorType. Port name is
	 *            optional
	 */
	public void sendMessage(ActorMsg msg) {
		curfsm.sendMessage(msg);
	}

	/**
	 * Sends a message to the specified actor address. Utilility routine to be
	 * used by the action specification in the state machine.
	 * 
	 * @param sig
	 * @param aa
	 *            The actor address that the message is sent to
	 */
	public void sendMessage(ActorMsg sig, ActorAddress aa) {
		curfsm.sendMessage(sig, aa);
	}

	/**
	 * Sends a message to an array of actor addresses. Utilility routine to be
	 * used by the action specification in the state machine.
	 * 
	 * @param sig
	 * @param actorAddresses
	 *            The array of actor address that the message is sent to
	 */
	public void sendMessage(ActorMsg sig, ActorAddress[] actorAddresses) {
		curfsm.sendMessage(sig, actorAddresses);
	}

	public void sendMessage(ActorMsg sig, Object[] actorAddresses) {
		curfsm.sendMessage(sig, actorAddresses);
	}

	public void sendMessage(String name, Object[] actorAddresses) {
		curfsm.sendMessage(name, actorAddresses);
	}

	public State getHistoryState() {
		return curfsm.getHistoryState(this);
	}

	public void setHistoryState(State state) {
		curfsm.setHistoryState(state);
	}

	public void setHistoryState() {
		curfsm.setHistoryState(curfsm.getCurrentState());
	}

	/**
	 * Log an error trace. The trace is send to Log4J, where appenders can be
	 * specified to log to screen, file etc.
	 * 
	 * @param str
	 *            is the trace information
	 */
	public void traceError(String str) {
		curfsm.getTrace().traceError(str);
	}

	/**
	 * Log a trace to thye transition. The trace is send to Log4J, where
	 * appenders can be specified to log to screen, file etc.
	 * 
	 * @param str
	 *            is the trace information
	 */
	public void traceTask(String str) {
		curfsm.getTrace().traceTask(str);
	}

	public ActorAddress getMyActorAddress() {
		return curfsm.getMyActorAddress();
	}

	public ActorAddress getParentActorAddress() {
		return curfsm.getContext().getParentAddress();
	}

	public void execTrans(ActorMsg sig, State st, StateMachine curfsm) {
	}
}
