/**
 * COOS - Connected Objects Operating System (www.connectedobjects.org).
 *
 * Copyright (C) 2009 Telenor ASA and Tellu AS. All rights reserved.
 * DO NOT ALTER OR REMOVE COPYRIGHT NOTICES OR THIS FILE HEADER.
 *
 * This library is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as published
 * by the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 *
 * You may also contact one of the following for additional information:
 * Telenor ASA, Snaroyveien 30, N-1331 Fornebu, Norway (www.telenor.no)
 * Tellu AS, Hagalokkveien 13, N-1383 Asker, Norway (www.tellu.no)
 */
package org.coos.javaframe;

import java.io.IOException;

import org.coos.actorframe.application.Container;
import org.coos.util.serialize.AFClassLoader;
import org.coos.javaframe.messages.ActorMsg;
import org.coos.javaframe.messages.Message;
import org.coos.actorframe.application.Session;
import org.coos.actorframe.threadpool.ThreadPool;
import org.coos.actorframe.threadpool.ThreadPoolRunnable;

/**
 * A multithreaded scheduler
 * 
 * @author Knut Eilif Husa, Tellu AS
 * 
 * 
 */
public class ThreadPooledScheduler extends SchedulerImpl {

	// The threadPool
	private ThreadPool tp = null;

	public ThreadPooledScheduler() {
		super();

	}

	/**
	 * The workerthread definition
	 */
	private class Worker implements Scheduler, ThreadPoolRunnable {
		private Schedulable sm;
		private ActorMsg msg;
		private TraceObject trace;

		public Worker(Schedulable sm, ActorMsg msg) {
			this.sm = sm;
			this.msg = msg;
		}

		public void run() {
			synchronized (sm) {
				sm.setScheduler(Worker.this);
				sm.processMessage(msg);
				if (sm.isReadyToBeDeleted()) {

					remove(sm.getMyActorAddress());

					if (isTraceOn()) {
						logger.log(TraceConstants.tlInfo, "execute: ActorStateMachine deleted: " + sm);
					}
				}

			}
		}

		public TraceObject getTraceObject() {
			return trace;
		}

		public void addRouter(Session domainSession) {
			ThreadPooledScheduler.this.addRouter(domainSession);
		}

		public void addSchedulable(Schedulable schedulable, ActorAddress adr) {
			ThreadPooledScheduler.this.addSchedulable(schedulable, adr);
		}

		public void addStateMachine(StateMachine sm, String type) {
			ThreadPooledScheduler.this.addStateMachine(sm, type);
		}

		public void clearLastMsgFromRouter() {
			ThreadPooledScheduler.this.clearLastMsgFromRouter();
		}

		public StateMachine createActor(ActorMsg sig, Schedulable curfsm) {
			return ThreadPooledScheduler.this.createActor(sig, curfsm);
		}

		public void destroyApp() {
			ThreadPooledScheduler.this.destroyApp();
		}

		public AFClassLoader getClassLoader() {
			return ThreadPooledScheduler.this.getClassLoader();
		}

		public String getName() {
			return ThreadPooledScheduler.this.getName();
		}

		public SchedulerData getSchedulerData() {
			return ThreadPooledScheduler.this.getSchedulerData();
		}

		public void interrupt() {
			ThreadPooledScheduler.this.interrupt();
		}

		public boolean output(ActorMsg sig, Schedulable schedulable) {
			return ThreadPooledScheduler.this.output(sig, schedulable);
		}

		public void postMessageToScheduler(Message message, Schedulable schedulable) throws IOException {
			ThreadPooledScheduler.this.postMessageToScheduler(message, schedulable);
		}

		public void remove(ActorAddress actorAddress) {
			ThreadPooledScheduler.this.remove(actorAddress);
		}

		public void removeRouter() {
			ThreadPooledScheduler.this.removeRouter();
		}

		public void setStopFlag() {
			ThreadPooledScheduler.this.setStopFlag();
		}

		public void unRegVisibleActor(ActorAddress address) {
			ThreadPooledScheduler.this.unRegVisibleActor(address);
		}

		public void upDateVisibleActors(StateMachine curfsm) {
			ThreadPooledScheduler.this.upDateVisibleActors(curfsm);
		}

		public boolean isTraceError() {
			return ThreadPooledScheduler.this.isTraceError();
		}

		public boolean isTraceOn() {
			return ThreadPooledScheduler.this.isTraceOn();
		}

		public void setTrace(boolean on) {
			ThreadPooledScheduler.this.setTrace(on);
		}

		public void setTraceError(boolean on) {
			ThreadPooledScheduler.this.setTraceError(on);
		}

		public void configure(String s) {
			ThreadPooledScheduler.this.configure(s);
		}

		public void setClassLoader(Container container) {
			ThreadPooledScheduler.this.setClassLoader(container);
		}

		public void setName(String id) {
			ThreadPooledScheduler.this.setName(id);
		}

		public void setSchedulerData(SchedulerData schedulerData) {
			ThreadPooledScheduler.this.setSchedulerData(schedulerData);
		}

		public void setThreads(int threads) {
			ThreadPooledScheduler.this.setThreads(threads);
		}

		public void start() {
			ThreadPooledScheduler.this.start();
		}

		public void setTraceObject(TraceObject trace) {
			this.trace = trace;
		}

		public void notifyScheduler() {
			ThreadPooledScheduler.this.notifyScheduler();
		}
	}

	/**
	 * This method redefines the message processing to use a threadpool
	 * 
	 * @param msg
	 *            - the message to be processed
	 * @param sm
	 *            - the schedulable to process on
	 */
	protected void processMessage(ActorMsg msg, Schedulable sm) {
		Worker worker = new Worker(sm, msg);
		tp.runRunnable(worker);
	}

	public void setThreads(int threads) {
		super.setThreads(threads);

	}

	public synchronized void start() {
		tp = new ThreadPool(getName(), getThreads());
		super.start();
	}
}
