/**
 * COOS - Connected Objects Operating System (www.connectedobjects.org).
 *
 * Copyright (C) 2009 Telenor ASA and Tellu AS. All rights reserved.
 * DO NOT ALTER OR REMOVE COPYRIGHT NOTICES OR THIS FILE HEADER.
 *
 * This library is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as published
 * by the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 *
 * You may also contact one of the following for additional information:
 * Telenor ASA, Snaroyveien 30, N-1331 Fornebu, Norway (www.telenor.no)
 * Tellu AS, Hagalokkveien 13, N-1383 Asker, Norway (www.tellu.no)
 */
package org.coos.javaframe;

/**
 * The class for the singular States of the StateMachines.
 * 
 * @author Dag Belsnes
 * @author Geir Melby, Tellu AS
 */
public class State {
	public CompositeState enclosingState = null; // the enclosing State of this
	// State
	private String stateName; // The symbolic name of this State

	/**
	 * State constructor with user supplied name.
	 * 
	 * @param sn
	 *            The name of this State.
	 * @param cs
	 *            The enclosing state
	 */
	public State(String sn, CompositeState cs) {
		stateName = sn;

		if (cs != null) {
			cs.children.put(sn, this); // adds this state to the encosing state
			// children
		}

		enclosingState = cs;
	}

	public void setStateName(String stateName) {
		this.stateName = stateName;
	}

	/**
	 * The name of the this State.
	 * 
	 * @return The symbolic name this State.
	 */
	public final String stateName() {
		return stateName;
	}

	/**
	 * The name of the this State.
	 * 
	 * @return The symbolic name this State.
	 */
	public final String getFullStateName() {
		String s = "";
		State st = enclosingState;

		if (st != null) {
			do {
				/*
				 * if (!s.equals("")) { s = s + "/"; }
				 */
				s = st.stateName + "/" + s;
				st = st.enclosingState;
			} while (st != null);
		}

		return s + stateName;
	}

	/**
	 * Update the enclosing states children table that contains reference to its
	 * childen states
	 * 
	 * @param sn
	 *            he id of the State to be looked for
	 * @param cs
	 *            The surrounding state that adds a State or CompositeState. NB!
	 *            The enclosing state is not set yest
	 */
	protected void updateChildren(String sn, CompositeState cs) {
		cs.children.put(sn, this); // add this state
	}

	/**
	 * Finds the chilren state based on its id
	 * 
	 * @param sn
	 *            The id of the State to be looked for
	 * @return Always null, because a State does not contain other states.
	 *         Normally the subtype CompositeState as overrided version of this
	 *         method.
	 */
	protected State findCurrentState(String sn) {
		return null;
	}

	/**
	 * Will be executed every time this State is entered. Default is empty.
	 * Note: not to be called from user code.
	 */
	public void entry(StateMachine curfsm) {
	}

	/**
	 * Will be executed every time this State is exited. Default is empty. Note:
	 * not to be called from user code.
	 */
	public void exit(StateMachine curfsm) {
	}

	// ####################################################################
	// Utility to be used within the transitions (see the methods
	// enterState(), execTrans(), and outofInnerCompositeState() of
	// CompositeState)

	/**
	 * Changes the current State of the StateMachine to this singular State.
	 * Note: a singular State do not have an entry index. Note: enterState
	 * should not be redefined for a singular State.
	 * 
	 * @param curfsm
	 *            The StateMachine that gets the new singular State.
	 */
	public void enterState(StateMachine curfsm) {
		curfsm.setNextState(this);
		curfsm.currentStateId = this.stateName();
		entry(curfsm);
	}

	/**
	 * Sets the next state. The framework will later call the enterState method
	 * 
	 * @param st
	 *            is the next state
	 * @param entryNo
	 *            entry number from 1...
	 * @param curfsm
	 *            is the current state machine state data
	 */
	public void nextState(State st, int entryNo, StateMachine curfsm) {
		curfsm.nextState = st;
		curfsm.entryNo = entryNo;
	}

	/**
	 * Sets the next state. The framework will later call the default enterState
	 * method
	 * 
	 * @param st
	 *            is the next state
	 * @param curfsm
	 *            is the current state machine state data
	 */
	public void nextState(State st, StateMachine curfsm) {
		curfsm.nextState = st;
		curfsm.entryNo = 0; // set deffault entry number;
	}

	public CompositeState getEnclosingState() {
		return enclosingState;
	}

	public boolean equals(Object o) {
		if (this == o)
			return true;
		if (!(o instanceof State)) {
			if (o instanceof String) {
				String name = (String) o;
				return stateName != null && stateName.equals(name);
			}
			return false;
		}

		State state = (State) o;

		if (enclosingState != null ? !enclosingState.equals(state.enclosingState) : state.enclosingState != null)
			return false;
		if (stateName != null ? !stateName.equals(state.stateName) : state.stateName != null)
			return false;

		return true;
	}

}
