/**
 * COOS - Connected Objects Operating System (www.connectedobjects.org).
 *
 * Copyright (C) 2009 Telenor ASA and Tellu AS. All rights reserved.
 * DO NOT ALTER OR REMOVE COPYRIGHT NOTICES OR THIS FILE HEADER.
 *
 * This library is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as published
 * by the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 *
 * You may also contact one of the following for additional information:
 * Telenor ASA, Snaroyveien 30, N-1331 Fornebu, Norway (www.telenor.no)
 * Tellu AS, Hagalokkveien 13, N-1383 Asker, Norway (www.tellu.no)
 */
package org.coos.javaframe;

import org.coos.actorframe.ActorReport;
import org.coos.javaframe.messages.ActorMsg;

import java.util.Enumeration;
import java.util.Hashtable;
import java.util.Vector;

/**
 * The ActorContext object holds information of the Actors context surroundings.
 * 
 * @author Knut Eilif Husa, Tellu AS
 */
public class ActorContext {
	private ActorAddress myAddress;
	private ActorAddress myParentAddress;
	private String historyStateId = "";
	private Hashtable requestors = new Hashtable();
	private Hashtable requested = new Hashtable();
	private Vector childrenRoles = new Vector();
	private Vector persistentChildren = new Vector();

	private Hashtable actorProperties;

	// used for report of status
	public ActorReport actorReport; // contains reports so fare
	ActorAddress senderOfReportRequest;
	private Vector creationOfChildren = new Vector();
	private ActorMsg rootPlayMsg = null;
	Vector tmpPorts = new Vector(); // tmp variable used when creating ports

	/**
	 * Creates a new instance of actor context for actor and stores the address
	 * of its parent actor.
	 * 
	 * @param myAddress
	 * @param myParentAddress
	 */
	public ActorContext(ActorAddress myAddress, ActorAddress myParentAddress) {
		this.myAddress = myAddress;
		this.myParentAddress = myParentAddress;
	}

	/**
	 * Creates a new context for an actor with its actor address as input
	 * 
	 * @param myAddress
	 *            is the actor address for the actor
	 */
	public ActorContext(ActorAddress myAddress) {
		this.myAddress = myAddress;
	}

	/**
	 * Checks the actor if it is persistent, meaning that it shall not be
	 * deleted if its container class (parent actor) still exist.
	 * 
	 * @return true if it is persistent
	 */

	/**
	 * Adds an requesting ActorAddress to this actor's context
	 * 
	 * @param key
	 *            is a name for the requestor role that can be used to identify
	 *            the requesting actor
	 * @param aa
	 *            the requesting ActorAddress
	 */
	public void addRequestorRole(String key, ActorAddress aa) {
		String tmp = key;

		if (tmp == null) {
			tmp = aa.toString();
		}

		if (requestors.containsKey(tmp)) {
			tmp = tmp + aa;
		}

		if (!requestors.containsKey(tmp)) {
			requestors.put(tmp, aa);
		}
	}

	public Hashtable getActorProperties() {
		return actorProperties;
	}

	public void setActorProperties(Hashtable actorProperties) {
		this.actorProperties = actorProperties;
	}

	/**
	 * Returns the requestors as a vector of actor addresses.
	 * 
	 * @return requestors.
	 */
	public Vector getRequestorRoles() {
		return getValues(requestors);
	}

	/**
	 * Checks whether an Actor represented by its ActorAddress is a requestor of
	 * this Actor.
	 * 
	 * @param aa
	 *            the ActorAddress to check.
	 * @return true if a requestor, otherwise false.
	 */
	public boolean requestorRolesContains(ActorAddress aa) {
		return requestors.contains(aa);
	}

	/**
	 * Checks whether the context contains no requestors or not.
	 * 
	 * @return true if empty
	 */
	public boolean requestorRolesIsEmpty() {
		return requestors.isEmpty();
	}

	/**
	 * Returns the ActorAddress of an Actor in this Actors requestor context
	 * corresponding to the actorID part of the Address.
	 * 
	 * @param actorID
	 *            The ID part of the Address
	 * @return aa the ActorAddress, null if not found
	 */
	public ActorAddress getRequestorRole(String actorID) {
		return getActorIDIgnoreCase(requestors.elements(), actorID);
	}

	/**
	 * Returns the RequestorRoles as a vector of ActorAddresses based on their
	 * requested type. The search is case-insensitive.
	 * 
	 * @param actorType
	 *            actor type to search for.
	 * @return RequestorRoles
	 */
	public Vector getRequestorRoles(String actorType) {
		return getActorTypesIgnoreCase(requestors.elements(), actorType);
	}

	/**
	 * Adds an requested ActorAddress to the context
	 * 
	 * @param aa
	 *            the requested ActorAddress
	 */
	public void addRequestedRole(String key, ActorAddress aa) {
		String tmp = key;

		if (tmp == null) {
			tmp = aa.toString();
		}

		if (requested.containsKey(tmp)) {
			tmp = tmp + aa;
		}

		if (!requested.containsKey(tmp)) {
			requested.put(tmp, aa);
		}
	}

	/**
	 * Returns the requested Actors as a vector of actor addresses.
	 * 
	 * @return requested Actors
	 */
	public Vector getRequestedRoles() {
		return getValues(requested);
	}

	/**
	 * Returns the RequestedRoles as a Vector of ActorAddresses based on their
	 * requested type. The search is case-insensitive.
	 * 
	 * @param actorType
	 * @return RequestedRoles
	 */
	public Vector getRequestedRoles(String actorType) {
		return getActorTypesIgnoreCase(requested.elements(), actorType);
	}

	/**
	 * Checks whether an Actor represented by its ActorAddress is a requested
	 * Actor of this Actor.
	 * 
	 * @param aa
	 *            the ActorAddress to check.
	 * @return true if a requested Actor, otherwise false.
	 */
	public boolean requestedRolesContains(ActorAddress aa) {
		return requested.contains(aa);
	}

	/**
	 * Checks whether the context contains no requested Roles.
	 * 
	 * @return true if empty
	 */
	public boolean requestedRolesIsEmpty() {
		return requested.isEmpty();
	}

	/**
	 * Returns the ActorAddress of an Actor in this Actor's requested context
	 * corresponding to the actorID part of the Address. The search is
	 * case-insensitive.
	 * 
	 * @param actorID
	 *            is the actorID part of the Address
	 * @return aa the ActorAddress, null if not found
	 */
	public ActorAddress getRequestedRole(String actorID) {
		return getActorIDIgnoreCase(requested.elements(), actorID);
	}

	/**
	 * Not used yet.
	 * 
	 * @param key
	 * @return Address to the actor if it exists, otherwise null
	 */
	public ActorAddress getConnection(Object key) {
		if (requested.get(key) != null) {
			return (ActorAddress) requested.get(key);
		} else if (requestors.get(key) != null) {
			return (ActorAddress) requestors.get(key);
		} else {
			return null;
		}
	}

	/**
	 * Not used yet
	 * 
	 * @return requested
	 */
	public Hashtable getConnections() {
		return requested;
	}

	/**
	 * Adds a children Role ActorAddress to the context.
	 * 
	 * @param aa
	 *            the ChildrenRole ActorAddress.
	 */
	public void addChildrenRole(ActorAddress aa) {
		if (!childrenRoles.contains(aa)) {
			childrenRoles.addElement(aa);
		}
	}

	/**
	 * Returns the ChildrenRoles as a vector of ActorAddresses.
	 * 
	 * @return ChildrenRoles
	 */
	public Vector getChildrenRoles() {
		Vector tmp = new Vector();
		addVector(tmp, childrenRoles);
		addVector(tmp, persistentChildren);

		return tmp;
	}

	/**
	 * Returns the ChildrenRoles as a vector of ActorAddresses
	 * 
	 * @return ChildrenRoles
	 */
	public Vector getNonPersistentChildrenRoles() {
		Vector tmp = new Vector();
		addVector(tmp, childrenRoles);

		return tmp;
	}

	/**
	 * Returns the ChildrenRoles as a vector of ActorAddresses based on their
	 * requested type. The search is case-insensitive.
	 * 
	 * @param actorType
	 *            type of actor to search for.
	 * @return ChildrenRoles
	 */
	public Vector getChildrenRoles(String actorType) { // NOTE changed by GM to
		// public

		Vector res = getActorTypesIgnoreCase(childrenRoles.elements(), actorType);
		addVector(res, getActorTypesIgnoreCase(persistentChildren.elements(), actorType));

		return res;
	}

	/**
	 * Adds a persistent Children role to this context. Persistent roles are
	 * never to be removed.
	 * 
	 * @param aa
	 *            the ActorAddress of the Children role.
	 */
	public void addPersistentChildrenRole(ActorAddress aa) {
		if (!persistentChildren.contains(aa)) {
			persistentChildren.addElement(aa);
		}
	}

	/**
	 * Checks whether a Role indicated by its ActorAddress is persistent or not.
	 * 
	 * @return true if persistent, otherwise false.
	 */
	public boolean isPersistentChildrenRole(ActorAddress aa) {
		return persistentChildren.contains(aa);
	}

	/**
	 * Checks whether an Actor represented by its ActorAddress is a childrenRole
	 * of this Actor.
	 * 
	 * @param aa
	 *            the ActorAddress to check.
	 * @return true if a childrenRole, otherwise false.
	 */
	public boolean childrenRolesContains(ActorAddress aa) {
		return childrenRoles.contains(aa) || persistentChildren.contains(aa);
	}

	/**
	 * Checks whether the context contains no childrenRoles.
	 * 
	 * @return true if empty.
	 */
	public boolean childrenRolesIsEmpty() {
		return childrenRoles.isEmpty() && persistentChildren.isEmpty();
	}

	/**
	 * Returns the ActorAddress of an Actor in this Actor's childrenRoles
	 * context corresponding to the actorID part of the ActorAddress.
	 * 
	 * @param actorID
	 *            The ID part of the ActorAddress.
	 * @return aa the ActorAddress, null if not found.
	 */
	public ActorAddress getChildrenRole(String actorID) {
		ActorAddress res = getActorIDIgnoreCase(childrenRoles.elements(), actorID);

		if (res != null) {
			return res;
		}

		// added by Viggo.
		return getActorIDIgnoreCase(persistentChildren.elements(), actorID);
	}

	/**
	 * Returns the ActorAddress of an Actor in this Actor's childrenRoles
	 * context corresponding to an ActorAddress without context information.
	 * 
	 * @param childRole
	 *            The ActorAddress without context information.
	 * @return aa the ActorAddress, null if not found.
	 */
	public ActorAddress getChildrenRoleIgnoreContext(ActorAddress childRole) {
		for (Enumeration e = childrenRoles.elements(); e.hasMoreElements();) {
			ActorAddress aa = (ActorAddress) e.nextElement();

			if (aa.equalsIgnoreContext(childRole)) {
				return aa;
			}
		}

		for (Enumeration e = persistentChildren.elements(); e.hasMoreElements();) {
			ActorAddress aa = (ActorAddress) e.nextElement();

			if (aa.equalsIgnoreContext(childRole)) {
				return aa;
			}
		}

		return null;
	}

	/**
	 * Contains checks whether an Actor represented by its ActorAddress is in
	 * the context of this Actor.
	 * 
	 * @param aa
	 *            the ActorAddress to check.
	 * @return true if present, otherwise false.
	 */
	public boolean contains(ActorAddress aa) {
		return (requestors.contains(aa) || requested.contains(aa) || childrenRoles.contains(aa) || persistentChildren
				.contains(aa));
	}

	/**
	 * Returns the number of Actors in this Actor's context.
	 * 
	 * @return the number of Actors in this Actor's context.
	 */
	public int size() {
		return requestors.size() + requested.size() + childrenRoles.size() + persistentChildren.size();
	}

	/**
	 * Checks whether this Actor's context is empty.
	 * 
	 * @return true if empty, otherwise false.
	 */
	public boolean isEmpty() {
		return (requestors.isEmpty() && requested.isEmpty() && childrenRoles.isEmpty() && persistentChildren.isEmpty());
	}

	/**
	 * Removes an Actor represented by its ActorAddress from this Actor's
	 * context.
	 * 
	 * @param aa
	 *            the ActorAddress of the Actor to remove.
	 * @return true if removed, otherwise false.
	 */
	public boolean remove(ActorAddress aa) {
		return (remove(requestors, aa) || remove(requested, aa) || childrenRoles.removeElement(aa) || persistentChildren
				.removeElement(aa));
	}

	/**
	 * Removes all actor instances represented by its ActorAddress from this
	 * Actor's context.
	 * 
	 * @param vaa
	 *            Vector of ActorAddresses to remove.
	 * @return boolean indicating whether any Vectors were removed.
	 */
	public boolean removeAll(Vector vaa) {
		boolean removed = false;

		for (int i = 0; i < vaa.size(); i++) {
			if (remove((ActorAddress) vaa.elementAt(i))) {
				removed = true;
			}
		}

		return removed;
	}

	/**
	 * Removes an Actor indicated by the actorID part of the ActorAddress from
	 * this Actor's context. The search is case-insensitive.
	 * 
	 * @param actorID
	 *            the actorID of the Actor to remove.
	 * @return true if removed, otherwise false.
	 */
	public boolean remove(String actorID) {
		ActorAddress aa = getActorIDIgnoreCase(requestors.elements(), actorID);

		if (aa != null) {
			return getValues(requestors).removeElement(aa);
		}

		aa = getActorIDIgnoreCase(requested.elements(), actorID);

		if (aa != null) {
			return getValues(requested).removeElement(aa);
		}

		aa = getActorIDIgnoreCase(childrenRoles.elements(), actorID);

		if (aa != null) {
			return childrenRoles.removeElement(aa);
		}

		aa = getActorIDIgnoreCase(persistentChildren.elements(), actorID);

		if (aa != null) {
			return persistentChildren.removeElement(aa);
		}

		return false;
	}

	/**
	 * Returns an Actor indicated by the actorID part of the ActorAddress from
	 * this Actor's context.
	 * 
	 * @param actorID
	 *            the actorID of the Actor to find.
	 * @return aa if found, otherwise null.
	 */
	public ActorAddress get(String actorID) {
		ActorAddress aa = getActorIDIgnoreCase(requestors.elements(), actorID);

		if (aa != null) {
			return aa;
		}

		aa = getActorIDIgnoreCase(requested.elements(), actorID);

		if (aa != null) {
			return aa;
		}

		aa = getActorIDIgnoreCase(childrenRoles.elements(), actorID);

		if (aa != null) {
			return aa;
		}

		aa = getActorIDIgnoreCase(persistentChildren.elements(), actorID);

		if (aa != null) {
			return aa;
		}

		return null;
	}

	/**
	 * Returns the ActorAddress of this context as a Collection.
	 * 
	 * @return collection of ActorAddress.
	 */

	/*
	 * public Collection values() { Vector tmp = new Vector();
	 * tmp.addAll(requestors.values()); tmp.addAll(requested.values());
	 * tmp.addAll(childrenRoles); tmp.addAll(persistentChildren); return tmp; }
	 */

	/**
	 * Returns this Actor's ActorAddress.
	 * 
	 * @return myAddress
	 */
	public ActorAddress getActorAddress() {
		return myAddress;
	}

	/**
	 * Returns the parent Actor of this role.
	 * 
	 * @return myParentAddress.
	 */
	public ActorAddress getParentAddress() {
		return myParentAddress;
	}

	/**
	 * Sets the parent address of the actor.
	 * 
	 * @param myParentAddress
	 *            is the parent address.
	 */
	public void setMyParentAddress(ActorAddress myParentAddress) {
		this.myParentAddress = myParentAddress;
	}

	/**
	 * Sets the actor address of this actor.
	 * 
	 * @param myAddress
	 *            is the actor address.
	 */
	public void setMyAddress(ActorAddress myAddress) {
		this.myAddress = myAddress;
	}

	/**
	 * A port was acknowledged, remove it from the tmp list and check whether
	 * there are more still ports which need acknowledgement.
	 * 
	 * @param portName
	 *            Name of the port to remove from the unacked list.
	 * @return boolean indicating whether there are more ports to be acked.
	 */
	public boolean allPortsAcked(String portName) {
		if ((tmpPorts == null) || tmpPorts.isEmpty()) {
			return true;
		}

		tmpPorts.removeElement(portName);

		return tmpPorts.isEmpty();
	}

	/**
	 * Returns the number of actors of actor type aa that are either persistent
	 * or not persistent children of this actor. The search is case-insensitive.
	 * 
	 * @param aa
	 *            is the actor type that specifies the search criteria.
	 * @return number of children maching the search criteria.
	 */
	public int sizeOfChildrenRoles(String aa) {
		int count = 0;

		for (int i = 0; i < childrenRoles.size(); i++) {
			ActorAddress actorAddress = (ActorAddress) childrenRoles.elementAt(i);

			if (actorAddress.getActorType().equals(aa)) {
				count = count + 1;
			}
		}

		for (int i = 0; i < persistentChildren.size(); i++) {
			ActorAddress actorAddress = (ActorAddress) persistentChildren.elementAt(i);

			if (actorAddress.getActorType().equals(aa)) {
				count = count + 1;
			}
		}

		return count;
	}

	public Vector getCreationOfChildren() {
		return creationOfChildren;
	}

	public void removeCreationChild(ActorAddress aa) {
		for (int i = 0; i < creationOfChildren.size(); i++) {
			ActorAddress refAA = (ActorAddress) creationOfChildren.elementAt(i);

			if (refAA.equals(aa)) {
				creationOfChildren.removeElement(refAA);

				return;
			}
		}
	}

	public ActorMsg getRootPlayMsg() {
		return rootPlayMsg;
	}

	public void setRootPlayMsg(ActorMsg rootPlayMsg) {
		this.rootPlayMsg = rootPlayMsg;
	}

	public Vector getTmpPorts() {
		return tmpPorts == null ? new Vector() : tmpPorts;
	}

	/**
	 * Copy those ports to tmp attribute that has an connector. Only these ports
	 * will receive an PortCreateMsg from the enclosing actor
	 * 
	 * @param tmpPorts
	 *            contains the port names
	 * @param connectors
	 *            conatains the connectors that later will be used to connect to
	 *            other ports
	 */
	public void setTmpPorts(Vector tmpPorts, Vector connectors) {
		this.tmpPorts = tmpPorts;
	}

	public String getHistoryStateId() {
		return historyStateId;
	}

	public void setHistoryStateId(String historyStateId) {
		this.historyStateId = historyStateId;
	}

	/**
	 * Perform a case insensitive search for actors of a given type.
	 * 
	 * @param searchIn
	 *            Enumeration to search in.
	 * @param actorType
	 *            actor type to search for.
	 * @return Vector of ActorAddresses of the found entries.
	 */
	private static Vector getActorTypesIgnoreCase(Enumeration searchIn, String actorType) {
		Vector res = new Vector();

		while (searchIn.hasMoreElements()) {
			ActorAddress aa = (ActorAddress) searchIn.nextElement();

			if (aa.getActorType().equals(actorType)) {
				res.addElement(aa);
			}
		}

		return res;
	}

	/**
	 * Perform a case insensitive search for actors of with a given ActorID.
	 * 
	 * @param searchIn
	 *            Enumeration to search in.
	 * @param actorID
	 *            actor ID to search for.
	 * @return ActorAddresses of the found entry.
	 */
	private static ActorAddress getActorIDIgnoreCase(Enumeration searchIn, String actorID) {
		while (searchIn.hasMoreElements()) {
			ActorAddress aa = (ActorAddress) searchIn.nextElement();

			if (aa.getActorID().equals(actorID)) {
				return aa;
			}
		}

		return null;
	}

	public boolean equals(Object obj) {
		ActorContext ac = (ActorContext) obj;

		if (!ac.myAddress.equals(this.myAddress)) {
			return false;
		}

		if (!ac.myParentAddress.equals(this.myParentAddress)) {
			return false;
		}

		if (!ac.requestors.equals(this.requestors)) {
			return false;
		}

		if (!ac.requested.equals(this.requested)) {
			return false;
		}

		if (!ac.childrenRoles.equals(this.childrenRoles)) {
			return false;
		}

		if (!ac.persistentChildren.equals(this.persistentChildren)) {
			return false;
		}

		if (!ac.creationOfChildren.equals(this.creationOfChildren)) {
			return false;
		}

		if (!ac.historyStateId.equals(this.historyStateId)) {
			return false;
		}

		if (!ac.senderOfReportRequest.equals(this.senderOfReportRequest)) {
			return false;
		}

		if (!ac.tmpPorts.equals(this.tmpPorts)) {
			return false;
		}

		return true;
	}

	private Vector getValues(Hashtable h1) {
		Vector v1 = new Vector();
		Enumeration e = h1.elements();

		while (e.hasMoreElements()) {
			Object o = e.nextElement();
			v1.addElement(o);
		}

		return v1;
	}

	private boolean remove(Hashtable h1, Object o) {
		Enumeration e = h1.keys();

		while (e.hasMoreElements()) {
			Object key = (Object) e.nextElement();
			Object value = h1.get(key);

			if (value.equals(o)) {
				h1.remove(key);

				return true;
			}
		}

		return false;
	}

	private void addVector(Vector v1, Vector v2) {
		for (int i = 0; i < v2.size(); i++) {
			Object o = (Object) v2.elementAt(i);
			v1.addElement(o);
		}
	}
}
