/**
 * COOS - Connected Objects Operating System (www.connectedobjects.org).
 *
 * Copyright (C) 2009 Telenor ASA and Tellu AS. All rights reserved.
 * DO NOT ALTER OR REMOVE COPYRIGHT NOTICES OR THIS FILE HEADER.
 *
 * This library is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as published
 * by the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 *
 * You may also contact one of the following for additional information:
 * Telenor ASA, Snaroyveien 30, N-1331 Fornebu, Norway (www.telenor.no)
 * Tellu AS, Hagalokkveien 13, N-1383 Asker, Norway (www.tellu.no)
 */
package org.coos.javaframe;

import org.coos.util.serialize.AFClassLoader;
import org.coos.util.serialize.AFSerializer;
import org.coos.util.serialize.ArrayHelper;
import org.coos.util.serialize.StringHelper;
import org.coos.actorframe.messages.AFConstants;
import org.coos.javaframe.messages.ActorAddressHelper;

import java.io.*;
import java.util.Hashtable;
import java.util.Vector;

/**
 * Contains the specification of inner parts of an actor. For each inner parts,
 * which again is an actor, the max number of instances that can be created and
 * the number of instanced that shall be created at creation of enclosing class.
 * 
 * @author Geir Melby, Tellu AS
 */
public class PartSpec implements AFSerializer {
	public String roleType;
	protected String[] roleNames;
	protected boolean credentialsRequired;
	protected boolean visible;
	protected int set = 0;
	protected int low = 1;
	protected int high = 1;
	protected int level = AFConstants.LEVEL_4; // level as default
	protected ActorAddress actorDomain;
	protected Hashtable portConnectors;
	protected String setId;
	private int traceLev = TraceConstants.tlWarn;
	private static String SET_ID = "set";
	// if different from null, it specfies the actor that implements this part.
	protected String bind;

	/**
	 * Empty constructor.
	 */
	public PartSpec() {
	}

	/**
	 * Specifies a part specification of only one part. Low and high are set to
	 * 1.
	 * 
	 * @param roleName
	 *            is the instance name of the part
	 * @param roleType
	 *            roleType is the type actor representing the part instance
	 */
	public PartSpec(String roleName, String roleType) {
		this.roleType = roleType;
		this.low = 1;
		this.high = 1;
		this.roleNames = new String[this.low];
		roleNames[0] = roleName;
		this.actorDomain = null;
		this.setId = SET_ID;
	}

	/**
	 * Specifies a part specification of only one part. Low and high are set to
	 * 1.
	 * 
	 * @param roleName
	 *            is the instance name of the part
	 * @param roleType
	 *            roleType is the type actor representing the part instance
	 * @param low
	 *            is the number of persistent instances that are created
	 *            initially. Each instance gets one of the name specified in
	 *            roleNames.
	 * @param high
	 *            is the max instance number allowed to be created
	 */
	public PartSpec(String roleName, String roleType, int low, int high) {
		this.roleType = roleType;
		this.low = low;
		this.high = high;

		if (low == 0) {
			this.roleNames = new String[1];
		} else {
			this.roleNames = new String[this.low];
		}
		roleNames[0] = roleName;
		this.actorDomain = null;
		this.setId = SET_ID;
	}

	/**
	 * Specifies a part instances where the instances are anonymous. The part
	 * instances get the actor names "roleTypeN" where N is a counter starting
	 * counting from 1.
	 * 
	 * @param roleNames
	 *            are the names that indentify the persistent state machine
	 *            instance created
	 * @param roleType
	 *            is the type actor representing the part instance
	 * @param low
	 *            low is the number of persistent instances that are created
	 *            initially. Each instance gets one of the name specified in
	 *            roleNames.
	 * @param high
	 *            s the max instance number allowed to be created
	 */
	public PartSpec(String[] roleNames, String roleType, int low, int high) {
		this.roleType = roleType;
		this.low = low;
		this.high = high;
		this.roleNames = roleNames;
		this.actorDomain = null;
		this.setId = SET_ID;
	}

	public int getLevel() {
		return level;
	}

	public void setLevel(int level) {
		this.level = level;
	}

	public void setRoleType(String roleType) {
		this.roleType = roleType;
	}

	public String getBind() {
		return bind;
	}

	public void setBind(String bind) {
		this.bind = bind;
	}

	public int getTraceLev() {
		return traceLev;
	}

	public void setTraceLev(int traceLev) {
		this.traceLev = traceLev;
	}

	public void setRoleNames(String[] roleNames) {
		this.roleNames = roleNames;
	}

	public void setLow(int low) {
		this.low = low;
	}

	public void setHigh(int high) {
		this.high = high;
	}

	/**
	 * Reads the actor domain address used when creating a parts
	 * 
	 * @return the actor domain address
	 */
	public ActorAddress getActorDomain() {
		return actorDomain;
	}

	public void setActorDomain(ActorAddress actorDomain) {
		this.actorDomain = actorDomain;
	}

	public boolean isCredentialsRequired() {
		return credentialsRequired;
	}

	public void setCredentialsRequired(boolean credentialsRequired) {
		this.credentialsRequired = credentialsRequired;
	}

	/**
	 * Check if a new instance are allowed to be created.
	 * 
	 * @param value
	 *            The current number of instances that already are created.
	 *            Normally is this value read from the two tables
	 *            "childrenRoles" and "persistentChildren" in the context
	 *            object.
	 * @return True if new instance can be created.
	 */
	public boolean exeedsCardinality(int value) {
		return (value < high);
	}

	public int getLow() {
		return low;
	}

	public int getHigh() {
		return high;
	}

	public String getRoleType() {
		return roleType;
	}

	public String[] getRoleNames() {
		if (roleNames == null) {
			roleNames = new String[0];
		}
		return roleNames;
	}

	public void setSetNo(int i) {
		this.set = i;
	}

	public int getSetNo() {
		return this.set;
	}

	public String getSetId() {
		return setId;
	}

	public void setSetId(String setId) {
		this.setId = setId;
	}

	public boolean isVisible() {
		return visible;
	}

	public void setVisible(boolean visible) {
		this.visible = visible;
	}

	public boolean equals(Object o) {
		if (this == o) {
			return true;
		}

		if (!(o instanceof PartSpec)) {
			return false;
		}

		final PartSpec part = (PartSpec) o;

		if ((roleType != null) ? (!roleType.equals(part.roleType)) : (part.roleType != null)) {
			return false;
		}

		return true;
	}

	public int hashCode() {
		return ((roleType != null) ? roleType.hashCode() : 0);
	}

	public String toString() {
		Vector inst = new Vector();

		if (roleNames != null) {
			for (int i = 0; i < roleNames.length; i++) {
				inst.addElement(roleNames[i]);
			}
		}

		String s = " Type: " + roleType + " Visible: " + isVisible() + " Min: " + low + " Max: " + high
				+ " Instances: " + inst.toString();

		if (actorDomain != null) {
			s += (" ActorDomain: " + actorDomain);
		}

		return s;
	}

	public String getPartDesc() {
		return this.getRoleType() + "(" + this.getLow() + "," + this.getHigh() + ")";
	}

	/**
	 * This function must implement the serialization of the object.
	 * 
	 * @return a byte array with the objects data
	 * @throws java.io.IOException
	 */
	public byte[] serialize() throws IOException {
		ByteArrayOutputStream bout = new ByteArrayOutputStream();
		DataOutputStream dout = new DataOutputStream(bout);
		dout.write(StringHelper.persist(roleType));
		dout.write(ArrayHelper.persist(roleNames));
		dout.writeInt(low);
		dout.writeInt(high);
		dout.writeInt(level);
		dout.writeInt(set);
		dout.write(ActorAddressHelper.persist(actorDomain));
		dout.write(StringHelper.persist(setId));
		dout.writeBoolean(credentialsRequired);
		dout.writeBoolean(visible);
		dout.writeInt(traceLev);
		dout.flush();

		return bout.toByteArray();
	}

	/**
	 * Use this function for resurrection of the object
	 * 
	 * @param data
	 *            The serialized data containing the object data
	 * @throws java.io.IOException
	 */
	public ByteArrayInputStream deSerialize(byte[] data, AFClassLoader cl) throws IOException {
		ByteArrayInputStream bin = new ByteArrayInputStream(data);
		DataInputStream din = new DataInputStream(bin);
		roleType = StringHelper.resurrect(din);
		roleNames = (String[]) ArrayHelper.resurrect(din);
		low = din.readInt();
		high = din.readInt();
		level = din.readInt();
		set = din.readInt();
		actorDomain = ActorAddressHelper.resurrect(din);
		setId = StringHelper.resurrect(din);
		credentialsRequired = din.readBoolean();
		visible = din.readBoolean();
		traceLev = din.readInt();
		return bin;
	}

	/**
	 * Check if an instance exists in the part spec
	 * 
	 * @param instance
	 *            is the name of the instance
	 * @return true if it exists
	 */
	public boolean containsInstance(String instance) {
		if (roleNames == null)
			return false;
		if (instance.lastIndexOf('/') > -1) {
			instance = instance.substring(instance.lastIndexOf('/') + 1);
		}
		for (int i = 0; i < roleNames.length; i++) {
			String roleName = roleNames[i];
			if (roleName.equals(instance)) {
				return true;
			}
		}
		return false;
	}
}
