/**
 * COOS - Connected Objects Operating System (www.connectedobjects.org).
 *
 * Copyright (C) 2009 Telenor ASA and Tellu AS. All rights reserved.
 * DO NOT ALTER OR REMOVE COPYRIGHT NOTICES OR THIS FILE HEADER.
 *
 * This library is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as published
 * by the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 *
 * You may also contact one of the following for additional information:
 * Telenor ASA, Snaroyveien 30, N-1331 Fornebu, Norway (www.telenor.no)
 * Tellu AS, Hagalokkveien 13, N-1383 Asker, Norway (www.tellu.no)
 */
package org.coos.javaframe;

import org.coos.actorframe.messages.AFConstants;
import org.coos.javaframe.messages.AFPropertyMsg;
import org.coos.javaframe.messages.ActorMsg;
import org.coos.javaframe.messages.JFConstants;

import java.util.Hashtable;
import java.util.Vector;

/**
 * @author Geir Melby, Tellu AS
 * 
 * 
 * 
 */
public class StateMachineTester implements AFConstants {
	StateMachine sm;
	// TraceObject to;
	TestScheduler scheduler;
	SchedulerData schedulerData;
	// ApplicationSpec appSpec;
	ActorAddress senderRole = new ActorAddress("test", "Tester");
	ActorAddress receiver;

	public StateMachineTester(StateMachine stateMachine, ActorAddress receiver, ApplicationSpec appSpec) {
		this.sm = stateMachine;
		this.sm.setTesting(true);
		this.receiver = receiver;
		sm.setMyActorId(receiver.getActorID());
		sm.setMyActorType(receiver.getActorType());
		schedulerData = new SchedulerData();
		scheduler = new TestScheduler(schedulerData);
		schedulerData.setApplicationSpec(appSpec);
		sm.setScheduler(scheduler);
		sm.init();
	}

	public boolean containsWaring() {
		Vector v = sm.getTrace().getWarnings();
		return v == null || v.isEmpty();
	}

	public boolean containsError() {
		Vector v = sm.getTrace().getErrors();
		return v == null || v.isEmpty();
	}

	public StateMachine getStateMacine() {
		return sm;
	}

	public void setAppSpec(ApplicationSpec appSpec) {
		// this.appSpec = appSpec;
		schedulerData.setApplicationSpec(appSpec);
	}

	public void processMessage(ActorMsg message, String portName) {
		ActorAddress receiver = sm.getMyActorAddress();
		receiver.setActorPort(portName);
		message.setReceiverRole(receiver);
		if (message.getSenderRole() == null) {
			message.setSenderRole(senderRole);
		}

		sm.processMessage(message);
	}

	public void processMessage(String message) {
		processMessage(new AFPropertyMsg(message));
	}

	public void processMessage(ActorMsg am) {
		if (am == null) {
			return;
		}

		if (am.getReceiverRole() == null) {
			am.setReceiverRole(sm.getMyActorAddress());
		}

		if (am.getSenderRole() == null) {
			am.setSenderRole(senderRole);
		}

		sm.processMessage(am);
	}

	public String getNextState() {
		return sm.getTrace().getNewState();
	}

	public String getCurrentState() {
		return sm.getCurrentState().stateName();
	}

	public boolean equalsState(String stateNasme) {
		return sm.getCurrentState().stateName().equals(stateNasme);
	}

	public boolean containsOutputMsg(String msgId) {
		Vector v = sm.getTrace().getOutputSignals();

		for (int i = 0; i < v.size(); i++) {
			ActorMsg msg = (ActorMsg) v.elementAt(i);

			if (msg.getSignalName().equals(msgId)) {
				return true;
			}
		}

		return false;
	}

	public int countOutputMsg() {
		return sm.getTrace().getOutputSignals().size();
	}

	public int countsOutputMsg(String msgId) {
		int no = 0;
		Vector v = sm.getTrace().getOutputSignals();

		for (int i = 0; i < v.size(); i++) {
			ActorMsg msg = (ActorMsg) v.elementAt(i);

			if (msg.getSignalName().equals(msgId)) {
				no++;
			}
		}

		return no;
	}

	public ActorMsg getOutputMsg(int index) {
		if (index < countOutputMsg()) {
			return (ActorMsg) sm.getTrace().getOutputSignals().elementAt(index);
		}
		return null;
	}

	public ActorMsg getOutputMsg(String msgId) {
		Vector v = sm.getTrace().getOutputSignals();

		for (int i = 0; i < v.size(); i++) {
			ActorMsg msg = (ActorMsg) v.elementAt(i);

			if (msg.getSignalName().equals(msgId)) {
				return msg;
			}
		}

		return null;
	}

	public boolean containsTimerMsg(String timerId) {
		Hashtable timers = sm.getActiveTimers();

		return ((timers != null) && (timers.get(timerId) != null));
	}

	/**
	 * Disable / enable the trace pint out
	 * 
	 * @param disabled
	 *            if true
	 */
	public void setPrintoutDisabled(boolean disabled) {
		sm.getTrace().setOutputDisabled(disabled);
		scheduler.setPrinting(!disabled);
	}

	public void sendTimer(String timerId) {
		AFPropertyMsg pm = new AFPropertyMsg(TIMER_MSG, true);
		pm.setProperty(JFConstants.TIMER_ID, timerId);
		processMessage(pm);
	}

	/**
	 * Creates an actor by using a RolePlay msg. This is the same situation as
	 * an actor creates an inner part based on a RoleRequest. The end state is
	 * idle if auccess
	 * 
	 * @return the state name of the state at end of the creation which is
	 *         normally is "idle"
	 */
	public String createActor() {
		AFPropertyMsg rrm = new AFPropertyMsg(AFConstants.ROLE_REQUEST_MSG, true);
		rrm.setProperty(ActorAddress.ROLE_ID, sm.getID());
		rrm.setProperty(ActorAddress.ROLE_TYPE, sm.getType());
		rrm.setSenderRole(senderRole);

		AFPropertyMsg pm = new AFPropertyMsg(AFConstants.ROLE_PLAY_MSG, true);
		pm.setProperty(ROLE_PLAY_MSG_RRM, rrm);
		pm.setProperty(ROLE_PLAY_MSG_PORTS, new Vector());
		processMessage(pm);

		return getCurrentState();
	}

	public void sendRoleRequestMsg(String roleId, String roleType) {
		// processMessage(new RoleRequestMsg(roleId, roleType));
		AFPropertyMsg pm = new AFPropertyMsg(ROLE_REQUEST_MSG, true);
		pm.setProperty(ROLE_REQUEST_MSG_ROLE_ID, roleId);
		pm.setProperty(ROLE_REQUEST_MSG_ROLE_TYPE, roleType);
		processMessage(pm);
	}

	public void sendRolePlayMsg(String[] portNames) {
		sendRolePlayMsg(sm.getID(), sm.getType(), getVector(portNames));
	}

	public void sendRolePlayMsg(String roleId, String roleType, Vector ports) {
		AFPropertyMsg rrm = new AFPropertyMsg(AFConstants.ROLE_REQUEST_MSG);
		rrm.setProperty(ActorAddress.ROLE_ID, roleId);
		rrm.setProperty(ActorAddress.ROLE_TYPE, roleType);
		rrm.setSenderRole(senderRole);

		AFPropertyMsg pm = new AFPropertyMsg(AFConstants.ROLE_PLAY_MSG, true);
		pm.setProperty("rrm", rrm);
		pm.setProperty(ROLE_PLAY_MSG_PORTS, ports);
		pm.setProperty(ROLE_PLAY_MSG_CONNECTORS, sm.actorSpec.getConnectorDesc(roleType, null));
		processMessage(pm);
	}

	public void sendRoleCreateMsg(String[] portNames) {
		Vector v = getVector(portNames);
		sendRoleCreateMsg(sm.getType(), v);
	}

	public void sendRoleCreateMsg(String enclosingType, Vector ports) {
		AFPropertyMsg pm = new AFPropertyMsg(ROLE_CREATE_MSG, true);
		pm.setReceiverRole(receiver);
		pm.setProperty(ROLE_CREATE_MSG_PORTS, ports);
		pm.setProperty(ROLE_CREATE_MSG_CONNECTORS, sm.actorSpec.getConnectorDesc(enclosingType, null));
		processMessage(pm);
	}

	public void sendRoleCreateMsg(String enclosingType) {
		AFPropertyMsg pm = new AFPropertyMsg(ROLE_CREATE_MSG, true);
		pm.setReceiverRole(receiver);
		ActorSpec enclosingActorSpec = schedulerData.getApplicationSpec().getActorSpec(enclosingType);
		pm.setProperty(ROLE_CREATE_MSG_PORTS, enclosingActorSpec.getConnectorDesc(sm.getType(), null));
		pm.setProperty(ROLE_CREATE_MSG_CONNECTORS, enclosingActorSpec.getConnectorDesc(sm.getType(), null));
		processMessage(pm);
	}

	private Vector getVector(String[] strings) {
		if (strings == null) {
			return new Vector();
		}

		Vector v = new Vector();

		for (int i = 0; i < strings.length; i++) {
			v.addElement(strings[i]);
		}

		return v;
	}

	/**
	 * Sets the current state machine to a specific state
	 * 
	 * @param stateName
	 *            is the name of the state
	 * @return true if the new state exists
	 */
	public boolean setNextState(String stateName) {
		State st = sm.getCompositeState().findCurrentState(stateName);

		if (st != null) {
			sm.setNextState(st);

			return true;
		}

		return false;
	}

	public ActorContext getActorContext() {
		return sm.context;
	}

	public void addActorSpec(ActorSpec actorSpec) {
		sm.getApplicationSpec().addActorSpec(actorSpec);
		sm.actorSpec = sm.getApplicationSpec().getClonedActorSpec(actorSpec.getActorType());
	}

}
