/**
 * COOS - Connected Objects Operating System (www.connectedobjects.org).
 *
 * Copyright (C) 2009 Telenor ASA and Tellu AS. All rights reserved.
 * DO NOT ALTER OR REMOVE COPYRIGHT NOTICES OR THIS FILE HEADER.
 *
 * This library is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as published
 * by the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 *
 * You may also contact one of the following for additional information:
 * Telenor ASA, Snaroyveien 30, N-1331 Fornebu, Norway (www.telenor.no)
 * Tellu AS, Hagalokkveien 13, N-1383 Asker, Norway (www.tellu.no)
 */
package org.coos.javaframe.messages;

import org.coos.javaframe.ActorAddress;
import org.coos.javaframe.LoggerFactory;
import org.coos.javaframe.TraceConstants;
import org.coos.util.serialize.AFClassLoader;
import org.coos.util.serialize.AFSerializer;

import java.io.*;
import java.util.Hashtable;
import java.util.Vector;

/**
 * ActorMsg is the general Message in Actorframe. It contains receiverRole and
 * the senderRole and a replyStack with information on for which path of Ports
 * the messages should visit if the message is to be replied to.
 * 
 */
public class ActorMsg extends Message implements AFSerializer {
	private ActorAddress receiverRole;
	private ActorAddress senderRole;
	private static int msgCounter = 0;
	private int msgRef;
	protected boolean frameworkMsg;
	private short msgType = MSG_TYPE_PROPERTY;
	public static final String BODY_PROPERTY = "Body";
	
	public static final short MSG_TYPE_PROPERTY = 0; // default type
	public static final short MSG_TYPE_TEST = 1; // test type
	public static final short MSG_TYPE_RESET = 2; // test type
	
	
	
	public ActorMsg() {
		receiverRole = null;
		senderRole = null;
		msgRef = ActorMsg.msgCounter++;
		frameworkMsg = false;
	}

    public static int getMsgCounter() {
        return msgCounter;
    }

    public static void setMsgCounter(int msgCounter) {
        ActorMsg.msgCounter = msgCounter;
    }

    public void assignId() {
		msgRef = ActorMsg.msgCounter++;
	}

	public int getMsgRef() {
		return msgRef;
	}

	public ActorMsg(ActorAddress aa) {
		this.receiverRole = aa;
		msgRef = ActorMsg.msgCounter++;
		frameworkMsg = false;
	}

	public ActorMsg(ActorMsg am) {
		this.receiverRole = am.getReceiverRole();
		this.senderRole = am.getSenderRole();
	}

	public short getMsgType() {
		return msgType;
	}

	public void setMsgType(short msgType) {
		this.msgType = msgType;
	}

	public ActorAddress getReceiverRole() {
		return receiverRole;
	}

	public ActorAddress getSenderRole() {
		return senderRole;
	}

	public void setSenderRole(String senderRole) {
		setSenderRole(new ActorAddress(senderRole));
	}

	public void setReceiverRole(ActorAddress recieverRole) {
		this.receiverRole = recieverRole;
	}

	public void setReceiverRole(String recieverRole) {
		setReceiverRole(new ActorAddress(recieverRole));
	}

	public void setMsgRef(int msgRef) {
		this.msgRef = msgRef;
	}

	public void setFrameworkMsg(boolean frameworkMsg) {
		this.frameworkMsg = frameworkMsg;
	}

	public void setSenderRole(ActorAddress senderRole) {
		this.senderRole = senderRole;
	}

	public boolean isFrameworkMsg() {
		return frameworkMsg;
	}

	public String getSignalName() {
		return getSignal(this.toString());
	}

	public boolean equals(int cmd) {
		return super.equals(new Integer(cmd));
	}

	public String messageContent() {
		String tmp = " MESSAGE: ID: " + msgRef + " NAME:" + getSignalName();

		if (receiverRole != null) {
			tmp = tmp + " RECEIVER:" + receiverRole.toString();
		} else {
			tmp = tmp + " RECEIVER: null";
		}

		if (senderRole != null) {
			tmp = tmp + " SENDER:" + senderRole.toString();
		} else {
			tmp = tmp + " SENDER: null";
		}

		return tmp + " CONTENT: ";
	}

	public static String getSignal(String s) {
		int start = 0;
		int end = 0;
		int length = s.length() - 1;

		for (int i = length; i >= 0; i--) {
			char c = s.charAt(i);

			if (c == '@') {
				end = i;
			}

			if (c == '.') {
				start = i + 1;
				i = 0;
			}
		}

		return s.substring(start, end);
	}

	public ActorMsg getCopy(AFClassLoader cl) {
		ActorMsg msg = null;

		try {
			byte[] data = serialize();
			String className = getClass().getName();
			Class myClass = null;

			if ((cl == null) || (cl.loadClass(className) == null)) {
				myClass = Class.forName(className);
			} else {
				myClass = cl.loadClass(className);
			}

			msg = (ActorMsg) myClass.newInstance();

			int msgNo = msg.msgRef;
			msg.deSerialize(data, cl);

			if (senderRole != null) {
				msg.senderRole = (ActorAddress) senderRole.clone();
			}

			if (receiverRole != null) {
				msg.receiverRole = (ActorAddress) receiverRole.clone();
			}

			msg.msgRef = msgNo;
		} catch (Exception e) {
			LoggerFactory.getLogger(this.getClass().getName()).log(TraceConstants.tlError, "Ignored Exception " + e);
		}

		return msg;
	}

	public String getMsgId() {
		throw new PropertyException();
	}

	public void setMsgId(String msgId) {
		throw new PropertyException();
	}
	
	public AFPropertyMsg setBody(Object body) {
		throw new PropertyException();
	}

	public void setProperties(Hashtable properties) {
		throw new PropertyException();
	}

	public Hashtable getProperty() {
		throw new PropertyException();
	}

	public Object getBody() {
		throw new PropertyException();
	}
	
	public Object getProperty(String name) {
		throw new PropertyException();
	}

	public String getString(String name) {
		throw new PropertyException();
	}

	public String getString(String name, String defaultValue) {
		throw new PropertyException();
	}

	public ActorAddress getActorAddress(String name) {
		throw new PropertyException();
	}

	public Vector getVector(String name) {
		throw new PropertyException();
	}

	public int getInt(String name) {
		throw new PropertyException();
	}

	public int getInt(String name, int defaultValue) {
		throw new PropertyException();
	}

	public boolean getBoolean(String name) {
		throw new PropertyException();
	}

	public AFPropertyMsg setByteArray(String name, byte[] array) {
		throw new PropertyException();
	}

	public byte[] getByteArray(String name) {
		throw new PropertyException();
	}

	public AFPropertyMsg setFloat(String name, float value) {
		throw new PropertyException();
	}

	public float getFloat(String name) {
		throw new PropertyException();
	}

	public AFPropertyMsg setDouble(String name, double value) {
		throw new PropertyException();
	}

	public double getDouble(String name) {
		throw new PropertyException();
	}

	public AFPropertyMsg setLong(String name, long value) {
		throw new PropertyException();
	}

	public long getLong(String name) {
		throw new PropertyException();
	}

	public long getLong(String name, long defaultValue) {
		throw new PropertyException();
	}

	public Object removeProperty(String name) {
		throw new PropertyException();
	}

	public AFPropertyMsg setProperty(String name, Object value) {
		throw new PropertyException();
	}

	public AFPropertyMsg setBoolean(String name, boolean value) {
		throw new PropertyException();
	}

	public AFPropertyMsg setInt(String name, int value) {
		throw new PropertyException();
	}

	public boolean hasProperty(String propertyName) {
		return false;
	}

	/**
	 * This function must implement the serialization of the object.
	 * 
	 * @return a byte array with the objects data
	 * @throws java.io.IOException
	 */
	public byte[] serialize() throws IOException {
		ByteArrayOutputStream bout = new ByteArrayOutputStream();
		DataOutputStream dout = new DataOutputStream(bout);
		dout.writeShort(msgType);
		if (msgType != MSG_TYPE_TEST) {
			dout.write(ActorAddressHelper.persist(receiverRole));
			dout.write(ActorAddressHelper.persist(senderRole));
			dout.writeInt(this.msgRef);
			dout.writeBoolean(this.frameworkMsg);
		}
		dout.flush();

		return bout.toByteArray();
	}

	public ByteArrayInputStream deSerialize(byte[] data, AFClassLoader cl) throws IOException {
		ByteArrayInputStream bin = new ByteArrayInputStream(data);
		DataInputStream din = new DataInputStream(bin);
		readInput(din, cl);

		return bin;
	}

	protected void readInput(DataInputStream din, AFClassLoader cl) throws IOException {
		msgType = din.readShort();
		if (msgType != MSG_TYPE_TEST) {
			receiverRole = ActorAddressHelper.resurrect(din);
			senderRole = ActorAddressHelper.resurrect(din);
			msgRef = din.readInt();
			frameworkMsg = din.readBoolean();
		}
	}

	public ActorMsg cloneMsg(AFClassLoader cl) {
		Class c = this.getClass();

		try {
			ActorMsg am = (ActorMsg) c.newInstance();

			if ((cl == null) || (cl.loadClass(c.getName()) == null)) {
				am.deSerialize(serialize(), null);
			} else {
				am.deSerialize(serialize(), cl);
			}

			return am;
		} catch (Exception e) {
			e.printStackTrace();

			return null;
		}
	}
}
