/**
 * COOS - Connected Objects Operating System (www.connectedobjects.org).
 *
 * Copyright (C) 2009 Telenor ASA and Tellu AS. All rights reserved.
 * DO NOT ALTER OR REMOVE COPYRIGHT NOTICES OR THIS FILE HEADER.
 *
 * This library is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as published
 * by the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 *
 * You may also contact one of the following for additional information:
 * Telenor ASA, Snaroyveien 30, N-1331 Fornebu, Norway (www.telenor.no)
 * Tellu AS, Hagalokkveien 13, N-1383 Asker, Norway (www.tellu.no)
 */
package org.coos.actorframe;

import org.coos.javaframe.ConnectorSpec;
import org.coos.javaframe.PartSpec;
import org.coos.util.serialize.AFClassLoader;
import org.coos.util.serialize.AFSerializer;
import org.coos.util.serialize.HashtableHelper;

import java.io.*;
import java.util.Enumeration;
import java.util.Hashtable;
import java.util.Vector;

/**
 * Contains the specification of inner parts of an actor. For each inner parts,
 * which again is an actor, the max number of instances that can be created and
 * the number of instanced that shall be created at creation of enclosing class.
 * 
 * @author Geir Melby, Tellu AS
 */
public class ActorPartSpec extends PartSpec implements AFSerializer {
	Hashtable actorProperties;

	/**
	 * Empty constructor.
	 */
	public ActorPartSpec() {
		portConnectors = new Hashtable();
	}

	/**
	 * Specifies a part instances where the instances are anonymous. The part
	 * instances get the actor names "roleTypeN" where N is a counter starting
	 * counting from 1.
	 * 
	 * @param roleNames
	 *            are the names that indentify the persistent state machine
	 *            instance created
	 * @param roleType
	 *            is the type actor representing the part instance
	 * @param low
	 *            low is the number of persistent instances that are created
	 *            initially. Each instance gets one of the name specified in
	 *            roleNames.
	 * @param high
	 *            s the max instance number allowed to be created
	 */
	public ActorPartSpec(String[] roleNames, String roleType, int low, int high) {
		super(roleNames, roleType, low, high);
	}

	/**
	 * Specifies a part instances where the instances are anonymous. The part
	 * instances get the actor names "roleTypeN" where N is a counter starting
	 * counting from 1.
	 * 
	 * @param roleNames
	 *            are the names that indentify the persistent state machine
	 *            instance created
	 * @param roleType
	 *            is the type actor representing the part instance
	 * @param low
	 *            low is the number of persistent instances that are created
	 *            initially. Each instance gets one of the name specified in
	 *            roleNames.
	 * @param high
	 *            s the max instance number allowed to be created
	 * @param actorProperties
	 *            string with actorid as key and hashtable of properties
	 */
	public ActorPartSpec(String[] roleNames, String roleType, int low, int high, Hashtable actorProperties) {
		super(roleNames, roleType, low, high);
		this.actorProperties = actorProperties;
	}

	public ActorPartSpec(String roleName, String roleType) {
		super(roleName, roleType, 1, 1);
	}

	public void setPortConnectors(Hashtable portConnectors) {
		this.portConnectors = portConnectors;
	}

	public String toString() {
		Vector inst = new Vector();

		if (roleNames != null) {
			for (int i = 0; i < roleNames.length; i++) {
				inst.addElement(roleNames[i]);
			}
		}

		String s = "Part: " + roleType + " Min: " + low + " Max: " + high + " Instances: " + inst.toString();

		if (portConnectors != null) {
			s += (" Ports: " + portConnectors.toString());
		}

		if (actorDomain != null) {
			s += (" ActorDomain: " + actorDomain);
		}

		return s;
	}

	public String getPartDesc() {
		return this.getRoleType() + "(" + this.getLow() + "," + this.getHigh() + ")";
	}

	public Hashtable getPorts() {
		return portConnectors;
	}

	/**
	 * Get the cloned Vector of ActorAddresses (connectors) for a given port.
	 * 
	 * @param portName
	 *            Portname to search for.
	 * @return Cloned Vector of ActorAddresses (connectors).
	 */
	public Vector getClonedConnectors(String portName) {
		Vector retconns = new Vector();
		Vector conns = getConnectors(portName);

		if (conns == null) {
			return retconns;
		}

		// clone to make sure no one rewrites it
		Enumeration enumer = conns.elements();

		while (enumer.hasMoreElements()) {
			ConnectorSpec cs = (ConnectorSpec) enumer.nextElement();
			retconns.addElement(cs.clone());
		}

		return retconns;
	}

	public boolean equals(Object o) {
		if (this == o) {
			return true;
		}

		if (!(o instanceof PartSpec)) {
			return false;
		}

		final ActorPartSpec part = (ActorPartSpec) o;

		/*
		 * if (!getUpdatedPortNames(part).isEmpty()) { return false; }
		 */

		if ((roleType != null) ? (!roleType.equals(part.roleType)) : (part.roleType != null)) {
			return false;
		}

		return true;
	}

	/**
	 * Get the connectors for a given port name in this PartSpec.
	 * 
	 * @param portName
	 *            name of the port to find connectors for.
	 * @return Vector of ActorAddresses representing the connectors of the given
	 *         port.
	 */
	public Vector getConnectors(String portName) {
		return (portConnectors != null) ? (Vector) portConnectors.get(portName) : new Vector();
	}

	/**
	 * Find the changed parts compared to other versions.
	 * 
	 * @param ps
	 * @return
	 * @deprecated not implemented yet.
	 */
	public Vector findUpdatedParts(ActorPartSpec ps) {
		// check if there are any new parts
		// todo: check for new parts.
		return new Vector();
	}

	/**
	 * Compare to other PartSpec whether the port configuration for this part is
	 * the same.
	 * 
	 * @param ps
	 *            PartSpec to compare with.
	 * @return Vector of portnames that differ within the PartSpecs.
	 */
	public Vector getUpdatedPortNames(ActorPartSpec ps) {
		Vector changedPorts = new Vector();
		Hashtable otherConn = ps.getPorts();
		if (otherConn == null) {
			return changedPorts;
		}
		// find new keys in otherConn
		Enumeration e1 = otherConn.keys();

		while (e1.hasMoreElements()) {
			String key = (String) e1.nextElement();

			if (!portConnectors.containsKey(key)) {
				changedPorts.addElement(key);
			}
		}

		// find new keys in portConn2
		Enumeration e2 = portConnectors.keys();

		while (e2.hasMoreElements()) {
			String key = (String) e2.nextElement();

			if (!otherConn.containsKey(key)) {
				changedPorts.addElement(key);
			}
		}

		// check the equality of the connectors
		e2 = portConnectors.keys();

		while (e2.hasMoreElements()) {
			String key = (String) e2.nextElement();

			if (changedPorts.contains(key)) {
				continue; // already there
			}

			// check connector equality
			Vector conn1 = (Vector) otherConn.get(key);
			Vector conn2 = getConnectors(key);

			if (conn1.size() != conn2.size()) {
				changedPorts.addElement(key); // sizes differ

				continue;
			}

			for (int i = 0; i < conn1.size(); i++) {
				if (!conn2.contains(conn1.elementAt(i))) {
					changedPorts.addElement(key);

					break;
				}
			}
		}

		return changedPorts;
	}

	/**
	 * This function must implement the serialization of the object.
	 * 
	 * @return a byte array with the objects data
	 * @throws java.io.IOException
	 */
	public byte[] serialize() throws IOException {
		ByteArrayOutputStream bout = new ByteArrayOutputStream();
		DataOutputStream dout = new DataOutputStream(bout);
		dout.write(super.serialize());
		dout.write(HashtableHelper.persist(portConnectors));
		dout.write(HashtableHelper.persist(actorProperties));
		dout.flush();

		return bout.toByteArray();
	}

	/**
	 * Use this function for resurrection of the object
	 * 
	 * @param data
	 *            The serialized data containing the object data
	 * @throws java.io.IOException
	 */
	public ByteArrayInputStream deSerialize(byte[] data, AFClassLoader cl) throws IOException {
		ByteArrayInputStream bin = super.deSerialize(data, cl);
		DataInputStream din = new DataInputStream(bin);
		portConnectors = HashtableHelper.resurrect(din, cl);
		actorProperties = HashtableHelper.resurrect(din, cl);

		return bin;
	}

	public Hashtable getActorProperties() {
		return actorProperties;
	}

	public void setActorProperties(Hashtable actorProperties) {
		this.actorProperties = actorProperties;
	}
}
